package hls.support.core.wechat.utils;

import com.jcraft.jsch.*;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.Properties;
import java.util.UUID;

/**
 * Created with IntelliJ IDEA.
 * User: yang
 * Date: 2017-11-09
 * Time: 20:41
 */
public class WechatFileUtil {

    private static Logger logger = LoggerFactory.getLogger(WechatFileUtil.class);

    public static boolean downloadSftpFile(String host,
                                           String port,
                                           String username,
                                           String password,
                                           String filePath,
                                           String savePath) {
        JSch jsch = new JSch();
        Session session = null;
        try {
            session = jsch.getSession(username, host, Integer.parseInt(port));
        } catch (JSchException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("jsch get session failed: {}", e);
            }
        }
        if (session == null) {
            return false;
        }
        session.setPassword(password);
        Properties sshConfig = new Properties();
        sshConfig.put("StrictHostKeyChecking", "no");
        session.setConfig(sshConfig);
        ChannelSftp sftp = null;
        InputStream inputStream = null;
        try {
            session.connect();
            Channel channel = session.openChannel("sftp");
            channel.connect();
            sftp = (ChannelSftp) channel;
            inputStream = sftp.get(filePath);
        } catch (JSchException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("jsch connect failed: {}", e);
            }
            session.disconnect();
            if (sftp != null) {
                sftp.exit();
            }
        } catch (SftpException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("sftp get file failed: {}", e);
            }
        }
        if (inputStream == null) {
            return false;
        }
        try {
            FileUtils.copyInputStreamToFile(inputStream, new File(savePath));
        } catch (IOException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("file download failed: {}", e);
            }
            return false;
        } finally {
            session.disconnect();
            if (sftp != null) {
                sftp.exit();
            }
        }
        return true;
    }

    public static boolean uploadBySftp(String host,
                                       String port,
                                       String username,
                                       String password,
                                       String savePath,
                                       String fileName,
                                       InputStream in) {
        JSch jsch = new JSch();
        Session session = null;
        try {
            session = jsch.getSession(username, host, Integer.parseInt(port));
        } catch (JSchException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("jsch get session failed: {}", e);
            }
        }
        if (session == null) {
            return false;
        }
        session.setPassword(password);
        Properties sshConfig = new Properties();
        sshConfig.put("StrictHostKeyChecking", "no");
        session.setConfig(sshConfig);
        ChannelSftp sftp = null;
        try {
            session.connect();
            Channel channel = session.openChannel("sftp");
            channel.connect();
            sftp = (ChannelSftp) channel;
        } catch (JSchException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("jsch connect failed: {}", e);
            }
            session.disconnect();
            if (sftp != null) {
                sftp.exit();
            }
        }
        if (fileName == null || fileName.trim().length() < 1) {
            fileName = UUID.randomUUID().toString().replaceAll("-", "");
        }
        if (sftp == null) {
            return false;
        }
        try {
            sftp.cd(savePath);
        } catch (Exception e) {
            try {
                sftp.mkdir(savePath);
                sftp.cd(savePath);
            } catch (SftpException e1) {
                logger.error("to server path({}) error:{}", savePath, e1);
            }
        }
        try {
            sftp.put(in, fileName);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            sftp.exit();
        }
        return false;
    }

    public static boolean saveFile(byte[] content, String filePath) throws IOException {
        File file = new File(filePath);
        if (file.exists() || file.isDirectory()) {
            throw new IOException("file exists or is a directory");
        } else {
            file.mkdir();
        }
        FileOutputStream out = new FileOutputStream(file);
        out.write(content);
        out.flush();
        out.close();
        return true;
    }

    public static boolean saveFile(InputStream in, String filePath) throws IOException {
        File file = new File(filePath);
        if (file.exists() || file.isDirectory()) {
            throw new IOException("file exists or is a directory");
        } else {
            file.mkdir();
        }
        FileUtils.copyInputStreamToFile(in, file);
        return true;
    }

    public static String getRandomFileName() {
        return StringUtils.removeAll(UUID.randomUUID().toString(), "-");
    }

    public static String getRandomFileName(String prefix) {
        return prefix
                + (prefix.charAt(prefix.length() - 1) == File.separatorChar ? "" : File.pathSeparatorChar)
                + getRandomFileName();
    }

    public static void downloadFile(String filePath, String fileName, HttpServletResponse response){
        File file = new File(filePath);
        if (!file.exists()){
            logger.error("target file does not exists.");
            response.setStatus(HttpStatus.SC_NO_CONTENT);
        }
        else {
            response.setContentType("application/octet-stream");
//          response.setContentType("application/force-download");// 设置强制下载不打开
//          response.setContentType("multipart/form-data");
            response.addHeader("Content-Disposition",
                    "attachment;fileName=" + fileName);// 设置文件名
            response.setContentLength((int)(FileUtils.sizeOf(file)));
            FileInputStream is = null;
            OutputStream os = null;
            try {
                is = new FileInputStream(file);
                os = response.getOutputStream();
                IOUtils.copy(is,os);
            } catch (Exception e) {
                logger.error("io transformation failed when download file.",e);
            } finally {
                IOUtils.closeQuietly(is);
                IOUtils.closeQuietly(os);
            }
        }
    }

}
