package hls.support.core.wechat.service.impl;

import com.github.pagehelper.PageHelper;
import com.hand.hap.core.IRequest;
import com.hand.hap.system.dto.ResponseData;
import com.hand.hap.system.service.impl.BaseServiceImpl;
import hls.support.core.wechat.dto.WechatAccount;
import hls.support.core.wechat.dto.WechatMember;
import hls.support.core.wechat.mapper.WechatAccountMapper;
import hls.support.core.wechat.mapper.WechatTagMapper;
import hls.support.core.wechat.utils.WeChatUtils;
import me.chanjar.weixin.common.exception.WxErrorException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.api.impl.WxCpServiceImpl;
import me.chanjar.weixin.cp.bean.WxCpTag;
import me.chanjar.weixin.cp.bean.WxCpUser;
import me.chanjar.weixin.cp.config.WxCpInMemoryConfigStorage;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import hls.support.core.wechat.dto.WechatTag;
import hls.support.core.wechat.service.IWechatTagService;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

@Service
@Transactional(rollbackFor = Exception.class)
public class WechatTagServiceImpl extends BaseServiceImpl<WechatTag> implements IWechatTagService {

    private static Logger logger = LoggerFactory.getLogger(WechatTagServiceImpl.class);
    @Autowired
    private WechatTagMapper wechatTagMapper;
    @Autowired
    private WechatAccountMapper wechatAccountMapper;
    @Value("#{configProperties['http.proxy.port']}")
    private Integer proxyPort;

    @Value("#{configProperties['http.proxy.host']}")
    private String proxyHost;

    @Override
    public List<WechatTag> queryTag(IRequest requestContext, WechatTag dto, int page, int pageSize) {
        PageHelper.startPage(page, pageSize);
        return wechatTagMapper.queryTag(dto);
    }

    /**
     * 同步标签数据
     */
    @Override
    public ResponseData refreshTag(IRequest reusetContext, Long accountId) {
        ResponseData responseData = new ResponseData(false);
        if (accountId == null) {
            responseData.setMessage("invalid params");
            logger.error("accountId is null when refresh tag datas!");
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(accountId);
        if (account == null) {
            responseData.setMessage("could not found target wechat account");
            logger.error("target wechat account can not be found!");
        }
        WxCpService wxCpService = WeChatUtils.initWxCpService(account.getAppId(), account.getSecret(), null, proxyHost, proxyPort);
        try {
            List<WxCpTag> wxCpTags = wxCpService.getTagService().listAll();
            WechatTag t = new WechatTag();
            t.setAccountId(accountId);
            List<WechatTag> wechatTags = wechatTagMapper.queryTag(t);
            List<String> tagIds = new ArrayList<>();
            /*删除已经不存在的数据*/
            for (WxCpTag tag : wxCpTags) {
                tagIds.add(tag.getId());
            }
            for (WechatTag tag : wechatTags) {
                if (tagIds.indexOf(tag.getTagId()) < 0) {
                    self().deleteByPrimaryKey(tag);
                }
            }
            /*插入或者更新数据*/
            for (WxCpTag wxCpTag : wxCpTags) {
                t.setTagId(wxCpTag.getId());
                t.setTagName(wxCpTag.getName());
                t.setAccountId(accountId);
                List<WechatTag> tags = wechatTagMapper.queryTag(t);
                if (CollectionUtils.isNotEmpty(tags)) {
                    t.setId(tags.get(0).getId());
                    wechatTagMapper.updateByPrimaryKey(t);
                } else {
                    wechatTagMapper.insertSelective(t);
                }
            }
            responseData.setSuccess(true);
        } catch (WxErrorException e) {
            logger.error("refresh tags data failed:", e);
            responseData.setMessage("refresh tags data failed.");
        }
        return responseData;
    }

    /**
     * 批量操作标签
     */
    @Override
    public Boolean batchUpdateTag(IRequest requestContext, List<WechatTag> tags) {
        for (WechatTag tag : tags) {
            switch (tag.get__status()) {
                case "add":
                    if (!self().addTag(requestContext, tag)) {
                        return false;
                    }
                    break;
                case "update":
                    if (!self().updateTag(requestContext, tag)) {
                        return false;
                    }
                    break;
                case "delete":
                    if (!self().deleteTag(requestContext, tag)) {
                        return false;
                    }
                    break;
                default:
                    logger.warn("__status is not set properly when batch update wechat tag.");
                    break;
            }
        }
        return true;
    }

    /**
     * 在标签中添加用户
     */
    @Override
    public boolean addTagUsers(IRequest iRequest, WechatTag tag) {
        String userIds = tag.getUserIds();
        if (StringUtils.isEmpty(userIds)) {
            return true;
        }
        if (tag.getAccountId() == null) {
            logger.error("accountId is null when add a wechat tag!");
            return false;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(tag.getAccountId());
        if (account == null) {
            logger.error("target wechat account can not be found!");
            return false;
        }
        WxCpService wxCpService = WeChatUtils.initWxCpService(account.getAppId(), account.getSecret(), null, proxyHost, proxyPort);
        try {
            List<String> userIdList = Arrays.asList(userIds.split(","));
            if (userIdList.get(userIdList.size() - 1).equals("")) {
                userIdList.remove(userIdList.size() - 1);
            }
            wxCpService.getTagService().addUsers2Tag(tag.getTagId(), userIdList, null);
        } catch (WxErrorException e) {
            logger.error("wxCpService add users to tag failed:", e);
            return false;
        }
        return true;
    }

    /**
     * 从标签中移除用户
     */
    @Override
    public boolean removeTagUsers(IRequest requestContext, WechatTag tag) {
        String userIds = tag.getUserIds();
        if (StringUtils.isEmpty(userIds)) {
            return true;
        }
        if (tag.getAccountId() == null) {
            logger.error("accountId is null when add a wechat tag!");
            return false;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(tag.getAccountId());
        if (account == null) {
            logger.error("target wechat account can not be found!");
            return false;
        }
        WxCpService wxCpService = WeChatUtils.initWxCpService(account.getAppId(), account.getSecret(), null, proxyHost, proxyPort);
        try {
            List<String> userIdList = Arrays.asList(userIds.split(","));
            if (userIdList.get(userIdList.size() - 1).equals("")) {
                userIdList.remove(userIdList.size() - 1);
            }
            wxCpService.getTagService().removeUsersFromTag(tag.getTagId(), userIdList);
        } catch (WxErrorException e) {
            logger.error("wxCpService delete delete users from tag failed:", e);
            return false;
        }
        return true;
    }

    /**
     * 删除标签
     */
    @Override
    public Boolean deleteTag(IRequest iRequest, WechatTag dto) {
        if (dto.getAccountId() == null) {
            logger.error("accountId is null when add a wechat tag!");
            return false;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(dto.getAccountId());
        if (account == null) {
            logger.error("target wechat account can not be found!");
            return false;
        }
        WxCpService wxCpService = WeChatUtils.initWxCpService(account.getAppId(), account.getSecret(), null, proxyHost, proxyPort);
        try {
            wxCpService.getTagService().delete(dto.getTagId());
        } catch (WxErrorException e) {
            logger.error("wxCpService delete a tag failed:", e);
            return false;
        }
        wechatTagMapper.deleteByPrimaryKey(dto);
        return true;
    }

    /**
     * 更新标签
     */
    @Override
    public Boolean updateTag(IRequest iRequest, WechatTag dto) {
        if (dto.getAccountId() == null) {
            logger.error("accountId is null when add a wechat tag!");
            return false;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(dto.getAccountId());
        if (account == null) {
            logger.error("target wechat account can not be found!");
            return false;
        }
        WxCpService wxCpService = WeChatUtils.initWxCpService(account.getAppId(), account.getSecret(), null, proxyHost, proxyPort);
        try {
            wxCpService.getTagService().update(dto.getTagId(), dto.getTagName());
        } catch (WxErrorException e) {
            logger.error("wxCpService update a tag failed:", e);
            return false;
        }
        wechatTagMapper.updateByPrimaryKeySelective(dto);
        return true;
    }

    /**
     * 创建标签
     */
    @Override
    public Boolean addTag(IRequest iRequest, WechatTag dto) {
        if (dto.getAccountId() == null) {
            logger.error("accountId is null when add a wechat tag!");
            return false;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(dto.getAccountId());
        if (account == null) {
            logger.error("target wechat account can not be found!");
            return false;
        }
        String tagId;
        WxCpService wxCpService = WeChatUtils.initWxCpService(account.getAppId(), account.getSecret(), null, proxyHost, proxyPort);
        try {
            tagId = wxCpService.getTagService().create(dto.getTagName());
        } catch (WxErrorException e) {
            logger.error("wxcpService create tag failed:", e);
            return false;
        }
        dto.setTagId(tagId);
        wechatTagMapper.insertSelective(dto);
        return true;
    }

}