package hls.support.core.wechat.service.impl;

import com.github.pagehelper.PageHelper;
import com.hand.hap.core.IRequest;
import com.hand.hap.system.dto.ResponseData;
import com.hand.hap.system.service.impl.BaseServiceImpl;
import hls.support.core.wechat.dto.*;
import hls.support.core.wechat.formbean.UserTag;
import hls.support.core.wechat.mapper.*;
import me.chanjar.weixin.common.exception.WxErrorException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.api.impl.WxCpServiceImpl;
import me.chanjar.weixin.cp.bean.WxCpUser;
import me.chanjar.weixin.cp.config.WxCpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.api.impl.WxMpServiceImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import hls.support.core.wechat.service.IWechatMemberService;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

@Service
@Transactional(rollbackFor = Exception.class)
public class WechatMemberServiceImpl extends BaseServiceImpl<WechatMember> implements IWechatMemberService {

    private static Logger logger = LoggerFactory.getLogger(WechatMemberServiceImpl.class);

    @Autowired
    private WechatMemberMapper wechatMemberMapper;

    @Autowired
    private WechatAccountMapper wechatAccountMapper;

    @Autowired
    private WechatMemberAdditionalInfoMapper wechatMemberAdditionalInfoMapper;

    @Autowired
    private WechatTagMapper wechatTagMapper;

    @Autowired
    private WechatMemberTagMapper wechatMemberTagMapper;

    @Override
    public List<WechatMember> queryMemberByDepart(IRequest iRequest, WechatMember wechatMember, int page, int pageSize) {
        PageHelper.startPage(page, pageSize);
        return wechatMemberMapper.queryMemberByDepart(wechatMember);
    }

    @Override
    public List<WechatMember> queryMemberByTag(IRequest iRequest, WechatMember wechatMember, int page, int pageSize) {
        PageHelper.startPage(page, pageSize);
        return wechatMemberMapper.queryMemberByTag(wechatMember);
    }

    @Override
    public List<WechatMember> queryMemberDetail(IRequest requestContext, WechatMember wechatMember, int page, int pageSize) {
        PageHelper.startPage(page, pageSize);
        return wechatMemberMapper.queryMemberDetail(wechatMember);
    }

    @Override
    public boolean batchUpdateMember(IRequest requestCtx, List<WechatMember> dto) {
        for (WechatMember member : dto) {
            switch (member.get__status()) {
                case "add":
                    if (!self().addMember(requestCtx, member)) {
                        return false;
                    }
                    break;
                case "update":
                    if (!self().updateMember(requestCtx, member)) {
                        return false;
                    }
                    break;
                case "delete":
                    if (!self().deleteMember(requestCtx, member)) {
                        return false;
                    }
                    break;
                default:
                    logger.warn("__status is not set properly when batch update wechat member.");
                    break;
            }
        }
        return true;
    }

    /**
     * 创建企业号用户
     */
    @Override
    public boolean addMember(IRequest requestCtx, WechatMember member) {
        if (member.getAccountId() == null) {
            logger.error("accountId is null when add a wechat tag!");
            return false;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(member.getAccountId());
        if (account == null) {
            logger.error("target wechat account can not be found!");
            return false;
        }
        Integer accountType = account.getAccountType();
        String appid = account.getAppId();
        String secret = account.getSecret();
        if (accountType == 1) {
            WxCpService wxCpService = new WxCpServiceImpl();
            WxCpInMemoryConfigStorage config = new WxCpInMemoryConfigStorage();
            config.setCorpSecret(secret);
            config.setCorpId(appid);
            wxCpService.setWxCpConfigStorage(config);
            WxCpUser user = new WxCpUser();
            user.setUserId(member.getUserId());
            user.setName(member.getName());
            String[] departmentIds = member.getDepartIds().split(",");
            Integer idsLength = departmentIds.length;
            if (departmentIds[idsLength - 1].equals("")) {
                idsLength -= 1;
            }
            Integer[] intIds = new Integer[idsLength];
            for (int i = 0; i < idsLength; i++) {
                intIds[i] = Integer.valueOf(departmentIds[i]);
            }
            user.setDepartIds(intIds);
            user.setEmail(member.getEmail());
            user.setMobile(member.getMobile());
            user.setPosition(member.getPosition());
            user.setTelephone(member.getTelephone());
            user.setEnglishName(member.getEnglishName());
            user.setEnable(member.getEnable());
            user.setAvatar(member.getAvatar());
            user.setHideMobile(member.getHideMobile());
            user.setStatus(member.getStatus());
            user.setIsLeader(member.getIsleader());
            switch (member.getGender()) {
                case WechatMember.GENDER_MALE_CODE:
                    user.setGender(WxCpUser.Gender.MALE);
                    break;
                case WechatMember.GENDER_FEMALE_CODE:
                    user.setGender(WxCpUser.Gender.FEMAIL);
                    break;
                default:
                    logger.warn("user's gender code is incorrect");
                    break;
            }
            try {
                wxCpService.getUserService().create(user);
            } catch (WxErrorException e) {
                logger.error("create qiye user failed: ", e);
                return false;
            }
            wechatMemberMapper.insertSelective(member);
            MemberAdditionalInfo memberAdditionalInfo = new MemberAdditionalInfo();
            memberAdditionalInfo.setIsleader(member.getIsleader());
            memberAdditionalInfo.setHideMobile(member.getHideMobile());
            memberAdditionalInfo.setMobile(member.getMobile());
            memberAdditionalInfo.setPosition(member.getPosition());
            memberAdditionalInfo.setTelephone(member.getTelephone());
            memberAdditionalInfo.setEnglishName(member.getEnglishName());
            memberAdditionalInfo.setMemberId(member.getMemberId());
            wechatMemberAdditionalInfoMapper.insertSelective(memberAdditionalInfo);
            return true;
        } else {
            logger.error("invalid account type.");
            return false;
        }
    }

    /**
     * 更新用户
     */
    @Override
    public boolean updateMember(IRequest requestCtx, WechatMember member) {
        if (member.getAccountId() == null) {
            logger.error("accountId is null when add a wechat tag!");
            return false;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(member.getAccountId());
        if (account == null) {
            logger.error("target wechat account can not be found!");
            return false;
        }
        Integer accountType = account.getAccountType();
        String appid = account.getAppId();
        String secret = account.getSecret();
        if (accountType == 1) {
            WxCpService wxCpService = new WxCpServiceImpl();
            WxCpInMemoryConfigStorage config = new WxCpInMemoryConfigStorage();
            config.setCorpSecret(secret);
            config.setCorpId(appid);
            wxCpService.setWxCpConfigStorage(config);
            WxCpUser user = new WxCpUser();
            user.setUserId(member.getUserId());
            user.setName(member.getName());
            String[] departmentIds = member.getDepartIds().split(",");
            Integer idsLength = departmentIds.length;
            if (departmentIds[idsLength - 1].equals("")) {
                idsLength -= 1;
            }
            Integer[] intIds = new Integer[idsLength];
            for (int i = 0; i < idsLength; i++) {
                intIds[i] = Integer.valueOf(departmentIds[i]);
            }
            user.setDepartIds(intIds);
            BeanUtils.copyProperties(member,user);
            switch (member.getGender()) {
                case WechatMember.GENDER_MALE_CODE:
                    user.setGender(WxCpUser.Gender.MALE);
                    break;
                case WechatMember.GENDER_FEMALE_CODE:
                    user.setGender(WxCpUser.Gender.FEMAIL);
                    break;
                default:
                    logger.warn("user's gender code is incorrect.");
                    break;
            }
            try {
                wxCpService.getUserService().update(user);
            } catch (WxErrorException e) {
                logger.error("update qiye user failed: ", e);
                return false;
            }
            wechatMemberMapper.updateByPrimaryKeySelective(member);
            MemberAdditionalInfo memberAdditionalInfo = new MemberAdditionalInfo();
            BeanUtils.copyProperties(member,memberAdditionalInfo);
            memberAdditionalInfo.setId(member.getAdditionalInfoId());
            wechatMemberAdditionalInfoMapper.updateByPrimaryKeySelective(memberAdditionalInfo);
        }
        /*非企业号只能更新备注*/
        else if (accountType == 2 || accountType == 3) {
            WxMpService wxMpService = new WxMpServiceImpl();
            WxMpInMemoryConfigStorage config = new WxMpInMemoryConfigStorage();
            config.setSecret(secret);
            config.setAppId(appid);
            wxMpService.setWxMpConfigStorage(config);
            try {
                wxMpService.getUserService().userUpdateRemark(member.getOpenId(), member.getRemark());
            } catch (WxErrorException e) {
                logger.error("update user remark failed:", e);
                return false;
            }
            MemberAdditionalInfo memberAdditionalInfo = new MemberAdditionalInfo();
            memberAdditionalInfo.setRemark(member.getRemark());
            memberAdditionalInfo.setId(member.getAdditionalInfoId());
            wechatMemberAdditionalInfoMapper.updateByPrimaryKeySelective(memberAdditionalInfo);
        } else {
            logger.error("invalid account type.");
            return false;
        }
        return true;
    }

    /**
     * 企业号删除用户
     */
    @Override
    public boolean deleteMember(IRequest requestCtx, WechatMember member) {
        if (member.getAccountId() == null) {
            logger.error("accountId is null when add a wechat tag!");
            return false;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(member.getAccountId());
        if (account == null) {
            logger.error("target wechat account can not be found!");
            return false;
        }
        Integer accountType = account.getAccountType();
        String appid = account.getAppId();
        String secret = account.getSecret();
        if (accountType == 1) {
            WxCpService wxCpService = new WxCpServiceImpl();
            WxCpInMemoryConfigStorage config = new WxCpInMemoryConfigStorage();
            config.setCorpSecret(secret);
            config.setCorpId(appid);
            wxCpService.setWxCpConfigStorage(config);
            try {
                wxCpService.getUserService().delete(member.getUserId());
            } catch (WxErrorException e) {
                logger.error("delete qiye user failed:", e);
                return false;
            }
            wechatMemberMapper.deleteByPrimaryKey(member);
            MemberAdditionalInfo additionalInfo = new MemberAdditionalInfo();
            additionalInfo.setId(member.getAdditionalInfoId());
            wechatMemberAdditionalInfoMapper.deleteByPrimaryKey(additionalInfo);
        } else {
            logger.error("invalid account type.");
            return false;
        }
        return true;
    }

    /**
     * 批量给用户添加或者移除标签
     */
    @Override
    public ResponseData batchUpdateUserTag(IRequest requestCtx, UserTag userTag) {
        ResponseData responseData = new ResponseData(false);
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(userTag.getAccountId());
        if (account == null) {
            logger.error("target account can not be found");
            responseData.setMessage("target account can not be found");
            return responseData;
        }
        int accountType = account.getAccountType();
        if (accountType == 2 || accountType == 3) {
            String secret = account.getSecret();
            String appid = account.getAppId();
            WechatTag tag = wechatTagMapper.selectByPrimaryKey(userTag.getTagId());
            if (!tag.getAccountId().equals(userTag.getAccountId())) {
                logger.error("tag not belong to the account");
                responseData.setMessage("tag not belong to the account");
                return responseData;
            }
            String[] memberIds = userTag.getMemberIds().split(",");
            Integer idsLength = memberIds.length;
            if (memberIds[idsLength - 1].equals("")) {
                idsLength -= 1;
            }
            String[] openIds = new String[idsLength];
            for (int i = 0; i < idsLength; i++) {
                Long id = Long.valueOf(memberIds[i]);
                WechatMember member = wechatMemberMapper.selectByPrimaryKey(id);
                if (!member.getAccountId().equals(userTag.getAccountId())) {
                    logger.error("user not belong to the account");
                    responseData.setMessage("user not belong to the account");
                    return responseData;
                }
                openIds[i] = member.getOpenId();
            }
            WxMpService wxMpService = new WxMpServiceImpl();
            WxMpInMemoryConfigStorage config = new WxMpInMemoryConfigStorage();
            config.setSecret(secret);
            config.setAppId(appid);
            wxMpService.setWxMpConfigStorage(config);
            switch (userTag.get__status()) {
                case "add":
                    try {
                        wxMpService.getUserTagService().batchTagging(Long.parseLong(tag.getTagId()), openIds);
                    } catch (WxErrorException e) {
                        logger.error("__status not properly set when batch add user tag");
                        responseData.setMessage("__status not properly set when batch add user tag");
                        return responseData;
                    }
                    for (String memberId : memberIds) {
                        WechatMemberTag memberTag = new WechatMemberTag();
                        memberTag.setTagId(userTag.getTagId());
                        memberTag.setMemberId(Long.valueOf(memberId));
                        wechatMemberTagMapper.insertSelective(memberTag);
                    }
                    break;
                case "delete":
                    try {
                        wxMpService.getUserTagService().batchUntagging(Long.parseLong(tag.getTagId()), openIds);
                    } catch (WxErrorException e) {
                        logger.error("__status not properly set when batch remove user tag");
                        responseData.setMessage("__status not properly set when batch remove user tag");
                        return responseData;
                    }
                    for (String memberId : memberIds) {
                        WechatMemberTag memberTag = new WechatMemberTag();
                        memberTag.setTagId(userTag.getTagId());
                        memberTag.setMemberId(Long.valueOf(memberId));
                        wechatMemberTagMapper.delete(memberTag);
                    }
                    break;
                default:
                    logger.error("invalid account type");
                    responseData.setMessage("invalid account type");
                    return responseData;
            }
        } else {
            logger.error("invalid account type");
            responseData.setMessage("invalid account type");
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }
}