package hls.support.core.wechat.service.impl;

import com.hand.hap.core.IRequest;
import com.hand.hap.system.dto.ResponseData;
import com.hand.hap.system.service.impl.BaseServiceImpl;
import hls.support.core.wechat.dto.*;
import hls.support.core.wechat.mapper.*;
import me.chanjar.weixin.common.exception.WxErrorException;
import me.chanjar.weixin.cp.api.impl.WxCpServiceImpl;
import me.chanjar.weixin.cp.bean.WxCpDepart;
import me.chanjar.weixin.cp.bean.WxCpTag;
import me.chanjar.weixin.cp.bean.WxCpUser;
import me.chanjar.weixin.cp.config.WxCpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.api.impl.WxMpServiceImpl;
import me.chanjar.weixin.mp.bean.result.WxMpUser;
import me.chanjar.weixin.mp.bean.result.WxMpUserList;
import me.chanjar.weixin.mp.bean.tag.WxUserTag;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import hls.support.core.wechat.service.IWechatAccountService;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

@Service
@Transactional(rollbackFor = Exception.class)
public class WechatAccountServiceImpl extends BaseServiceImpl<WechatAccount> implements IWechatAccountService {

    private static Logger logger = LoggerFactory.getLogger(WechatAccountServiceImpl.class);
    @Autowired
    private WechatAccountMapper wechatAccountMapper;
    @Autowired
    private WechatDepartmentMapper wechatDepartmentMapper;
    @Autowired
    private WechatTagMapper wechatTagMapper;
    @Autowired
    private WechatMemberMapper wechatMemberMapper;
    @Autowired
    private WechatMemberDepartmentMapper wechatMemberDepartmentMapper;
    @Autowired
    private WechatMemberTagMapper wechatMemberTagMapper;
    @Autowired
    private WechatMemberAdditionalInfoMapper memberAdditionalInfoMapper;
    @Autowired
    private WechatEnterpriseAppMapper enterpriseAppMapper;
    @Autowired
    private AppDepartmentMapper appDepartmentMapper;
    @Autowired
    private AppMemberMapper appMemberMapper;

    @Value("#{configProperties['http.proxy.port']}")
    private Integer proxyPort;

    @Value("#{configProperties['http.proxy.host']}")
    private String proxyHost;

    private WechatAccount validateParam(Long accountId) {
        if (accountId != null) {
            WechatAccount account = wechatAccountMapper.selectByPrimaryKey(accountId);
            if (account != null) {
                return account;
            }
            logger.error("could not found target wechat account");
        }
        logger.error("accountId is null.");
        return null;
    }

    /**
     * 清楚共同数据
     *
     * @param accountId
     */
    private void clearCommonData(Long accountId) {
        /*根据accountId删除当前公众号的所有用户信息*/
        WechatMember m = new WechatMember();
        m.setAccountId(accountId);
        List<WechatMember> wechatMembers = wechatMemberMapper.select(m);
        if (CollectionUtils.isNotEmpty(wechatMembers)){
            for (WechatMember member : wechatMembers) {
                MemberAdditionalInfo memberAdditionalInfo = new MemberAdditionalInfo();
                memberAdditionalInfo.setMemberId(member.getMemberId());
                memberAdditionalInfoMapper.delete(memberAdditionalInfo);
                wechatMemberMapper.delete(member);
            }
        }
        /*根据标签id删除标签和用户对应关系*/
        WechatTag wt = new WechatTag();
        wt.setAccountId(accountId);
        List<WechatTag> wts = wechatTagMapper.select(wt);
        if (CollectionUtils.isNotEmpty(wts)){
            for (WechatTag wechatTag : wts) {
                WechatMemberTag wmt = new WechatMemberTag();
                wmt.setTagId(wechatTag.getId());
                wechatMemberTagMapper.delete(wmt);
            }
        }
        /*删除本地该公众号号下的所有标签*/
        wechatTagMapper.delete(wt);
    }

    /**
     * 同步企业号数据
     *
     * @param account
     * @return
     */
    private ResponseData synchronizeCpData(WechatAccount account) {
        ResponseData responseData = new ResponseData(false);
        /*根据accountId查询该企业号下的应用list*/
        WechatEnterpriseApp wa = new WechatEnterpriseApp();
        wa.setAccountId(account.getAccountId());
        List<WechatEnterpriseApp> enterpriseApps = enterpriseAppMapper.select(wa);

        WxCpInMemoryConfigStorage config = new WxCpInMemoryConfigStorage();
        config.setCorpId(account.getAppId());
        config.setCorpSecret(account.getSecret());
        if (proxyHost!=null){
            config.setHttpProxyHost(proxyHost);
        }
        if (proxyPort!=null){
            config.setHttpProxyPort(proxyPort);
        }
        WxCpServiceImpl wxCpService = new WxCpServiceImpl();
        wxCpService.setWxCpConfigStorage(config);

        /*获取微信服务器上该企业号的所有部门和标签列表*/
        List<WxCpDepart> departList;
        List<WxCpTag> tagList;
        try {
            departList = wxCpService.getDepartmentService().listAll();
            tagList = wxCpService.getTagService().listAll();
        } catch (WxErrorException e) {
            logger.error("fetch department or tag list from wechat server failed:", e);
            responseData.setMessage("fetch department or tag list from wechat server failed");
            return responseData;
        }

        /*根据部门id删除部门和用户对应关系*/
        WechatDepartment wd = new WechatDepartment();
        wd.setAccountId(account.getAccountId());
        List<WechatDepartment> wds = wechatDepartmentMapper.select(wd);
        if (CollectionUtils.isNotEmpty(wds)){
            for (WechatDepartment wechatDepartment : wds) {
                WechatMemberDepartment wmd = new WechatMemberDepartment();
                wmd.setDepartmentId(wechatDepartment.getId());
                wechatMemberDepartmentMapper.delete(wmd);
            }
        }
        /*删除本地该企业号下所有部门数据*/
        wechatDepartmentMapper.delete(wd);

        /*插入新的部门数据*/
        List<WechatDepartment> wechatDepartments = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(departList)){
            for (WxCpDepart depart : departList) {
                WechatDepartment department = new WechatDepartment();
                department.setAccountId(account.getAccountId());
                department.setDepartmentId(depart.getId());
                department.setDepartmentName(depart.getName());
                department.setParentId(depart.getParentId());
                department.setDepartmentOrder(depart.getOrder());
                wechatDepartmentMapper.insertSelective(department);
                wechatDepartments.add(department);
            }
        }

        /*根据新的部门数据查询新的用户数据，并插入新的用户数据和新的用户部门对应关系*/
        if (CollectionUtils.isNotEmpty(wechatDepartments)){
            for (WechatDepartment department : wechatDepartments) {
                List<WxCpUser> users;
                try {
                    users = wxCpService.getUserService().listByDepartment(department.getDepartmentId(), false, 0);
                } catch (WxErrorException e) {
                    logger.error("fetch users data which based on departmentId from wechat server failed: ", e);
                    responseData.setMessage("fetch users data which based on departmentId from wechat server failed.");
                    return responseData;
                }
                if (CollectionUtils.isNotEmpty(users)){
                    for (WxCpUser user : users) {
                        WechatMember wechatMember = new WechatMember();
                        wechatMember.setUserId(user.getUserId());
                        wechatMember.setAccountId(account.getAccountId());
                        WechatMember member = new WechatMember();
                        List<WechatMember> members = wechatMemberMapper.select(wechatMember);

                        /*使用userid查询用户是否已经在数据库中*/
                        if (CollectionUtils.isEmpty(members)) {
                            /*插入用户到数据库*/
                            BeanUtils.copyProperties(user, member);
                            //member.setGender(user.getGender());
                            member.setAccountId(account.getAccountId());
                            wechatMemberMapper.insertSelective(member);

                            /*插入用户附加信息到数据库*/
                            MemberAdditionalInfo memberAdditionalInfo = new MemberAdditionalInfo();
                            BeanUtils.copyProperties(user, memberAdditionalInfo);
                            memberAdditionalInfo.setMemberId(member.getMemberId());
                            memberAdditionalInfoMapper.insertSelective(memberAdditionalInfo);
                        } else {
                            if (members.size() > 1) {
                                logger.warn("duplicate user_id data in table wechat_member!");
                                responseData.setMessage("duplicate user_id data in table wechat_member!");
                                return responseData;
                            }
                            /*如果用户已经存在，获取当前查询到的用户*/
                            member = members.get(0);
                        }

                        /*插入用户和部门对应关系到数据库*/
                        WechatMemberDepartment memberDepartment = new WechatMemberDepartment();
                        memberDepartment.setDepartmentId(department.getId());
                        memberDepartment.setMemberId(member.getMemberId());
                        wechatMemberDepartmentMapper.insertSelective(memberDepartment);
                    }
                }
            }
        }

        /*插入新的标签数据*/
        List<WechatTag> wechatTags = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(tagList)){
            for (WxCpTag tag : tagList) {
                WechatTag wechatTag = new WechatTag();
                wechatTag.setAccountId(account.getAccountId());
                wechatTag.setTagName(tag.getName());
                wechatTag.setTagId(tag.getId());
                wechatTagMapper.insertSelective(wechatTag);
                wechatTags.add(wechatTag);
            }
        }

        /*建立用户标签对应关系*/
        if (CollectionUtils.isNotEmpty(wechatTags)){
            for (WechatTag wechatTag : wechatTags) {
                /*根据标签id获取标签下的用户*/
                List<WxCpUser> users;
                try {
                    users = wxCpService.getTagService().listUsersByTagId(wechatTag.getTagId());
                } catch (WxErrorException e) {
                    logger.error("fetch users data which based on tagId failed: ", e);
                    responseData.setMessage("fetch users data which based on tagId failed.");
                    return responseData;
                }
                /*根据查询到的用户建立用户标签对应关系*/
                if (CollectionUtils.isNotEmpty(users)) {
                    for (WxCpUser user : users) {
                        WechatMember member = new WechatMember();
                        member.setUserId(user.getUserId());
                        List<WechatMember> members = wechatMemberMapper.select(member);
                        if (!CollectionUtils.isEmpty(members)) {
                            if (members.size() > 1) {
                                logger.warn("duplicate user_id data in table wechat_member!");
                                responseData.setMessage("duplicate user_id data in table wechat_member!");
                                return responseData;
                            }
                            WechatMemberTag memberTag = new WechatMemberTag();
                            member = wechatMemberMapper.select(member).get(0);
                            memberTag.setTagId(wechatTag.getId());
                            memberTag.setMemberId(member.getMemberId());
                            wechatMemberTagMapper.insertSelective(memberTag);
                        }
                    }
                }
            }
        }

        /*循环遍历当前企业号的所有应用，建立应用和部门、应用和用户的权限对应关系*/
        if (CollectionUtils.isNotEmpty(enterpriseApps)) {
            for (WechatEnterpriseApp app : enterpriseApps) {
                config.setCorpSecret(app.getSecret());
                wxCpService.setWxCpConfigStorage(config);
                EnterpriseAppMember am = new EnterpriseAppMember();
                am.setAppId(app.getId());
                appMemberMapper.delete(am);
                EnterpriseAppDepartment ad = new EnterpriseAppDepartment();
                ad.setAppId(app.getId());
                appDepartmentMapper.delete(ad);
                /*使用应用的token获取可见部门*/
                List<WxCpDepart> departs;
                List<WxCpUser> users;
                try {
                    departs = wxCpService.getDepartmentService().listAll();
                } catch (WxErrorException e) {
                    logger.error("fetch department data by current account's apps failed," +
                            " the error app's agentId is " + app.getAgentId(), e);
                    responseData.setMessage("fetch department data by current account's apps failed," +
                            " the error app's agentId is " + app.getAgentId());
                    return responseData;
                }
                if (CollectionUtils.isNotEmpty(departs)) {
                    for (WxCpDepart depart : departs) {
                        /*在数据库中找到对应的部门，并建立应用和部门的可见性对应关系*/
                        WechatDepartment department = new WechatDepartment();
                        department.setAccountId(account.getAccountId());
                        department.setDepartmentId(depart.getId());
                        List<WechatDepartment> wechatDepartmentList = wechatDepartmentMapper.select(department);
                        if (!CollectionUtils.isEmpty(wechatDepartmentList)) {
                            if (wechatDepartmentList.size() > 1) {
                                logger.warn("duplicate account_id & department_id data in table wechat_department!");
                                responseData.setMessage("duplicate account_id & department_id data in table wechat_department!");
                                return responseData;
                            }
                            WechatDepartment wechatDepartment = wechatDepartmentList.get(0);
                            EnterpriseAppDepartment appDepartment = new EnterpriseAppDepartment();
                            appDepartment.setAppId(app.getId());
                            appDepartment.setDepartmentId(wechatDepartment.getId());
                            appDepartmentMapper.insertSelective(appDepartment);
                            /*查找部门下的用户，并遍历用户与应用建立可见性对应关系*/
                            try {
                                users = wxCpService.getUserService()
                                        .listSimpleByDepartment(depart.getId(), true, 0);
                            } catch (WxErrorException e) {
                                logger.error("fetch users data by current app's department failed," +
                                        " the error app's agentId is {} ", app.getAgentId(), e);
                                responseData.setMessage("fetch users data by current app's department failed," +
                                        " the error app's agentId is " + app.getAgentId());
                                return responseData;
                            }
                            if (CollectionUtils.isNotEmpty(users)){
                                for (WxCpUser user : users) {
                                    WechatMember wechatMember = new WechatMember();
                                    wechatMember.setUserId(user.getUserId());
                                    List<WechatMember> members = wechatMemberMapper.select(wechatMember);
                                    if (!CollectionUtils.isEmpty(members)) {
                                        if (members.size() > 1) {
                                            logger.warn("duplicate user_id data in table wechat_member!");
                                            responseData.setMessage("duplicate user_id data in table wechat_member!");
                                            return responseData;
                                        }
                                        WechatMember member = members.get(0);
                                        EnterpriseAppMember appMember = new EnterpriseAppMember();
                                        appMember.setMemberId(member.getMemberId());
                                        appMember.setAppId(app.getId());
                                        if (CollectionUtils.isEmpty(appMemberMapper.select(appMember)))
                                            appMemberMapper.insertSelective(appMember);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        responseData.setSuccess(true);
        return responseData;
    }

    /**
     * 同步非企业号数据
     *
     * @param account
     * @return
     */
    private ResponseData synchronizeMpData(WechatAccount account) {
        ResponseData responseData = new ResponseData(false);
        List<WxUserTag> tagList;
        WxMpUserList userList;
        WxMpInMemoryConfigStorage config = new WxMpInMemoryConfigStorage();
        config.setAppId(account.getAppId());
        config.setSecret(account.getSecret());
        if (proxyHost!=null){
            config.setHttpProxyHost(proxyHost);
        }
        if (proxyPort!=null){
            config.setHttpProxyPort(proxyPort);
        }
        WxMpService wxMpService = new WxMpServiceImpl();
        wxMpService.setWxMpConfigStorage(config);
        try {
            tagList = wxMpService.getUserTagService().tagGet();
            userList = wxMpService.getUserService().userList(null);
        } catch (WxErrorException e) {
            logger.error("fetch tag or user list from wechat server failed: ", e);
            responseData.setMessage("fetch tag or user list from wechat server failed");
            return responseData;
        }
        /*插入新的标签数据*/
        for (WxUserTag tag : tagList) {
            WechatTag wechatTag = new WechatTag();
            wechatTag.setAccountId(account.getAccountId());
            wechatTag.setTagName(tag.getName());
            wechatTag.setTagId(String.valueOf(tag.getId()));
            wechatTagMapper.insertSelective(wechatTag);
        }
        List<String> openIds = userList.getOpenids();
        if (CollectionUtils.isNotEmpty(openIds)) {
            /*插入新的用户信息*/
            for (String openId : openIds) {
                WxMpUser user;
                try {
                    user = wxMpService.getUserService().userInfo(openId);
                } catch (WxErrorException e) {
                    logger.error("fetch user info by openid failed,", e);
                    responseData.setMessage("fetch user info by openid failed.");
                    return responseData;
                }
                WechatMember member = new WechatMember();
                MemberAdditionalInfo additionalInfo = new MemberAdditionalInfo();
                member.setOpenId(user.getOpenId());
                member.setGender(String.valueOf(user.getSexId()));
                member.setAccountId(account.getAccountId());
                member.setName(user.getNickname());
                member.setAvatar(user.getHeadImgUrl());
                wechatMemberMapper.insertSelective(member);
                BeanUtils.copyProperties(user, additionalInfo);
                additionalInfo.setGroupid(user.getGroupId());
                additionalInfo.setMemberId(member.getMemberId());
                memberAdditionalInfoMapper.insertSelective(additionalInfo);
                /*插入新的用户标签对应数据*/
                Long[] tagIds = user.getTagIds();
                for (Long tagId : tagIds) {
                    WechatTag wechatTag = new WechatTag();
                    wechatTag.setTagId(String.valueOf(tagId));
                    wechatTag.setAccountId(account.getAccountId());
                    List<WechatTag> tags = wechatTagMapper.select(wechatTag);
                    if (CollectionUtils.isNotEmpty(tags)) {
                        WechatMemberTag memberTag = new WechatMemberTag();
                        memberTag.setMemberId(member.getMemberId());
                        memberTag.setTagId(tags.get(0).getId());
                        wechatMemberTagMapper.insertSelective(memberTag);
                    }
                }
            }
        }
        responseData.setSuccess(true);
        return responseData;
    }

    @Override
    public ResponseData synchronizeData(IRequest requestContext, Long accountId) {
        ResponseData responseData = new ResponseData(false);
        WechatAccount account = validateParam(accountId);
        if (account != null) {
            clearCommonData(accountId);
            int accountType = account.getAccountType();
            if (accountType == WechatAccount.CORP) {
                return synchronizeCpData(account);
            } else if (accountType == WechatAccount.SUBS || accountType == WechatAccount.SERV) {
                return synchronizeMpData(account);

            } else {
                logger.error("account type is incorrect");
                responseData.setMessage("account type is incorrect");
                return responseData;
            }
        } else {
            responseData.setMessage("target weixin account can not be found");
            return responseData;
        }

    }

}