package com.hand.hls.utils;

import com.hand.hap.core.components.ApplicationContextHelper;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;

import javax.crypto.Cipher;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * RSA 加密解码工具类 ,公钥私钥的获取策略为先从Redis中获取,如果不存在则创建后写入到Redis中,多实例共享
 *
 * @author qiang.chen04@hand-china.com
 */
public class RSACoder {

    public static final String KEY_ALGORITHM = "RSA";
    public static final String SIGNATURE_ALGORITHM = "MD5withRSA";
    private static final Logger logger = LoggerFactory.getLogger(RSACoder.class);
    private static final String PUBLIC_KEY = "RSAPublicKey";
    private static final String PRIVATE_KEY = "RSAPrivateKey";

    /**
     * 存储在Redis中的公钥Key
     */
    public static final String REDIS_PUBLIC_KEY = "hls:ras:" + PUBLIC_KEY;
    /**
     * 存储在Redis中的私钥Key
     */
    public static final String REDIS_PRIVATE_KEY = "hls:ras:" + PRIVATE_KEY;

    private static final RedisTemplate redisTemplate;


    static {
        redisTemplate = ApplicationContextHelper.getApplicationContext().getBean(RedisTemplate.class);
    }


    public static byte[] decryptBASE64(String key) {
        return Base64.decodeBase64(key);
    }

    public static String encryptBASE64(byte[] bytes) {
        return Base64.encodeBase64String(bytes);
    }

    /**
     * 用私钥对信息生成数字签名
     *
     * @param data       加密数据
     * @param privateKey 私钥
     */
    public static String sign(byte[] data, String privateKey) throws Exception {
        // 解密由base64编码的私钥
        byte[] keyBytes = decryptBASE64(privateKey);
        // 构造PKCS8EncodedKeySpec对象
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
        // KEY_ALGORITHM 指定的加密算法
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        // 取私钥匙对象
        PrivateKey priKey = keyFactory.generatePrivate(pkcs8KeySpec);
        // 用私钥对信息生成数字签名
        Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
        signature.initSign(priKey);
        signature.update(data);
        return encryptBASE64(signature.sign());
    }

    /**
     * 校验数字签名
     *
     * @param data      加密数据
     * @param publicKey 公钥
     * @param sign      数字签名
     * @return 校验成功返回true 失败返回false
     */
    public static boolean verify(byte[] data, String publicKey, String sign) throws Exception {
        // 解密由base64编码的公钥
        byte[] keyBytes = decryptBASE64(publicKey);
        // 构造X509EncodedKeySpec对象
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keyBytes);
        // KEY_ALGORITHM 指定的加密算法
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        // 取公钥匙对象
        PublicKey pubKey = keyFactory.generatePublic(keySpec);
        Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
        signature.initVerify(pubKey);
        signature.update(data);
        // 验证签名是否正常
        return signature.verify(decryptBASE64(sign));
    }

    public static byte[] decryptByPrivateKey(byte[] data, String key) throws Exception {
        // 对密钥解密
        byte[] keyBytes = decryptBASE64(key);
        // 取得私钥
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key privateKey = keyFactory.generatePrivate(pkcs8KeySpec);
        // 对数据解密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.DECRYPT_MODE, privateKey);
        return cipher.doFinal(data);
    }


    public static CharSequence encryptByPrivateKey(CharSequence rawPassword) throws Exception {
        String privateKey = getPrivateKey();
        String s = rawPassword.toString();
        byte[] bytes = decryptByPrivateKey(s, privateKey);
        return new String(bytes);
    }

    /**
     * 解密<br>
     * 用私钥解密
     */
    public static byte[] decryptByPrivateKey(String data, String key)
            throws Exception {
        return decryptByPrivateKey(decryptBASE64(data), key);
    }

    /**
     * 解密<br>
     * 用公钥解密
     */
    public static byte[] decryptByPublicKey(byte[] data, String key) throws Exception {
        // 对密钥解密
        byte[] keyBytes = decryptBASE64(key);
        // 取得公钥
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key publicKey = keyFactory.generatePublic(x509KeySpec);
        // 对数据解密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.DECRYPT_MODE, publicKey);
        return cipher.doFinal(data);
    }

    /**
     * 加密<br>
     * 用公钥加密
     */
    public static byte[] encryptByPublicKey(String data, String key) throws Exception {
        // 对公钥解密
        byte[] keyBytes = decryptBASE64(key);
        // 取得公钥
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key publicKey = keyFactory.generatePublic(x509KeySpec);
        // 对数据加密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.ENCRYPT_MODE, publicKey);
        return cipher.doFinal(data.getBytes());
    }

    /**
     * 加密<br>
     * 用私钥加密
     */
    public static byte[] encryptByPrivateKey(byte[] data, String key) throws Exception {
        // 对密钥解密
        byte[] keyBytes = decryptBASE64(key);
        // 取得私钥
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key privateKey = keyFactory.generatePrivate(pkcs8KeySpec);
        // 对数据加密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.ENCRYPT_MODE, privateKey);
        return cipher.doFinal(data);
    }

    /**
     * 取得私钥
     */
    public static String getPrivateKey() {
        final Object privateKey = redisTemplate.opsForValue().get(REDIS_PRIVATE_KEY);
        if (Objects.nonNull(privateKey)) {
            return privateKey.toString();
        }
        Map<String, String> rasKeyPair = refreshRedisPublicRSAKey();
        return rasKeyPair.get(REDIS_PRIVATE_KEY);
    }

    /**
     * 取得公钥
     */
    public static String getPublicKey() {
        final Object publicKey = redisTemplate.opsForValue().get(REDIS_PUBLIC_KEY);
        if (Objects.nonNull(publicKey)) {
            return publicKey.toString();
        }
        Map<String, String> rasKeyPair = refreshRedisPublicRSAKey();
        return rasKeyPair.get(REDIS_PUBLIC_KEY);
    }


    /**
     * 刷新RedisKey
     */
    public static Map<String, String> refreshRedisPublicRSAKey() {
        Map<String, String> resultMap = new HashMap<>();
        KeyPairGenerator keyPairGen = null;
        try {
            keyPairGen = KeyPairGenerator.getInstance(KEY_ALGORITHM);
        } catch (NoSuchAlgorithmException e) {
            logger.error("fail to init KeyPairGenerator");
            resultMap.put(REDIS_PUBLIC_KEY,"");
            resultMap.put(REDIS_PRIVATE_KEY,"");
            return resultMap;
        }
        keyPairGen.initialize(1024);
        KeyPair keyPair = keyPairGen.generateKeyPair();
        // 公钥
        final String publicKey = encryptBASE64(keyPair.getPublic().getEncoded());
        // 私钥
        final String privateKey = encryptBASE64(keyPair.getPrivate().getEncoded());
        redisTemplate.opsForValue().set(REDIS_PUBLIC_KEY, publicKey);
        redisTemplate.opsForValue().set(REDIS_PRIVATE_KEY, privateKey);
        resultMap.put(REDIS_PUBLIC_KEY,publicKey);
        resultMap.put(REDIS_PRIVATE_KEY,privateKey);
        return resultMap;
    }

//    public static void main(String[] args) throws Exception {
//        String publicKey = getPublicKey();
//        String privateKey = getPrivateKey();
//
//        System.out.println(keyMap);
//        System.out.println("-----------------------------------");
//        System.out.println(publicKey);
//        System.out.println("-----------------------------------");
//        System.out.println(privateKey);
//        System.out.println("-----------------------------------");
//        byte[] encryptByPrivateKey = encryptByPrivateKey("123456".getBytes(),privateKey);
//        byte[] encryptByPublicKey = encryptByPublicKey("123456",publicKey);
//        System.out.println(new String(encryptByPrivateKey));
//        System.out.println("-----------------------------------");
//        System.out.println(new String(encryptByPublicKey));
//        System.out.println("-----------------------------------");
//        String sign = sign(encryptByPrivateKey,privateKey);
//        System.out.println(sign);
//        System.out.println("-----------------------------------");
//        boolean verify = verify(encryptByPrivateKey,publicKey,sign);
//        System.out.println(verify);
//        System.out.println("-----------------------------------");
//        byte[] decryptByPublicKey = decryptByPublicKey(encryptByPrivateKey,publicKey);
//        byte[] decryptByPrivateKey = decryptByPrivateKey(encryptByPublicKey,privateKey);
//        System.out.println(new String(decryptByPublicKey));
//        System.out.println("-----------------------------------");
//        System.out.println(new String(decryptByPrivateKey));
//
//    }

}
