package com.hand.app.sms.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.chinapay.comm.Constants;
import com.google.gson.reflect.TypeToken;
import com.hand.app.sms.dto.ResponseData;
import com.hand.app.sms.dto.SmsResponse;
import com.hand.app.sms.service.IHclcSmsLogService;
import com.hand.app.sms.service.IHclcSmsService;
import com.hand.app.sms.utils.DateUtil;
import com.hand.app.sms.utils.JsonHelper;
import com.hand.app.sms.utils.Md5;
import com.hand.app.sms.utils.http.*;
import com.hand.app.zhongDengWang.dto.HlsWsRequests;
import com.hand.hap.core.IRequest;
import com.hand.hap.intergration.dto.HapInterfaceHeader;
import com.hand.hap.intergration.service.IHapInterfaceHeaderService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;


/**
 * Copyright (C) Hand Business Consulting Services
 * AllRights Reserved
 *
 * @author: Eugene Song
 * @date: 2023/6/12
 * @description:
 */
@Service
public class HclcSmsServiceImpl implements IHclcSmsService {

    @Autowired
    private IHapInterfaceHeaderService headerService;
    @Autowired
    private IHclcSmsLogService hclcSmsLogService;

    //用户认证接口信息
    public static final String SYS_NAME = "HCLC_SMS_API";
    public static final String SEND_SMS = "sendSms";


    /*事务状态*/
    public static final String tranSuccess = "SUCCESS";
    public static final String transFail = "FAIL";


    private Logger logger = LoggerFactory.getLogger(getClass());

    public String getApiUrl(String sysName, String apiName) {
        //初始化接口密钥参数信息
        HapInterfaceHeader headerAndLineDTO = headerService.getHeaderAndLine(sysName, apiName);
        if (headerAndLineDTO == null) {
            logger.info("headerAndLineDTO is null， apiName:{}, sysName:{}", apiName, sysName);
            return null;
        }
        return headerAndLineDTO.getDomainUrl() + headerAndLineDTO.getIftUrl();
    }


    public boolean isNumeric(String str) {
        try {
            Long.parseLong(str);
            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    @Override
    public JSONObject sendSms(IRequest iRequest, JSONObject params) {
        logger.info("=============start sendSms==================");
        JSONObject result = new JSONObject();

        //appId && secretKey 通过社内 传入，以后修改的时候 只需要修改 pkg 就可以 不用修改java代码重启
        if (Objects.isNull(params.get("appId"))) {
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "请求必须参数appId缺失，请检查！");
            return result;
        }

        if (Objects.isNull(params.get("secretKey"))) {
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "请求必须参数secretKey缺失，请检查！");
            return result;
        }

        if (Objects.isNull(params.get("customSmsId"))) {
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "请求必须参数customSmsId缺失，请检查！");
            return result;
        }

        if (Objects.isNull(params.get("content"))) {
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "请求必须参数wcontent缺失，请检查！");
            return result;
        }
        if (Objects.isNull(params.get("mobiles"))) {
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "请求必须参数mobiles缺失，请检查！");
            return result;
        }

        if (!isNumeric(params.get("mobiles").toString())) {
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "非法的电话号码！");
            return result;
        }


        String appId = params.get("appId").toString();
        // 密钥
        String secretKey = params.get("secretKey").toString();
        // 时间戳
        String timestamp = DateUtil.toString(new Date(), "yyyyMMddHHmmss");
        // 签名
        String sign = Md5.md5((appId + secretKey + timestamp).getBytes());
        // 接口地址(中台接口平台定义)
        String sendSmsApiUrl = getApiUrl(SYS_NAME, SEND_SMS);

        logger.info("token url: {}", sendSmsApiUrl);
        if (sendSmsApiUrl != null) {
            String request = "appId=" + appId + "&timestamp=" + timestamp + "&sign=" + sign + "&mobiles" + params.get("mobiles").toString() + "&content=" + params.get("content").toString() + "&customSmsId=" + params.get("customSmsId").toString();
            HlsWsRequests log = hclcSmsLogService.createLog(iRequest, "hclc_sms", sendSmsApiUrl, request, Long.parseLong(params.get("mobiles").toString()));
            result = setSms(iRequest, appId, sign, timestamp, sendSmsApiUrl, params.get("content").toString(), params.get("mobiles").toString(), params.get("customSmsId").toString(), null, null, log);
        } else {
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "接口平台URL未定义，请检查！");
            return result;
        }

        logger.info("=============end sendSms==================");
        return result;
    }

    /**
     * 发送批次短信
     */
    private JSONObject setSms(IRequest iRequest, String appId, String sign, String timestamp, String sendSmsApiUrl, String content, String mobiles, String customSmsId, String extendedCode, String timerTime, HlsWsRequests log) {
        JSONObject result = new JSONObject();
        logger.info("=============start setSms==================");
        Map<String, String> params = new HashMap<String, String>();
        try {
            params.put("appId", appId);
            params.put("sign", sign);
            params.put("timestamp", timestamp);
            params.put("mobiles", mobiles);
            params.put("content", URLEncoder.encode(content, "utf-8"));
            if (customSmsId != null) {
                params.put("customSmsId", customSmsId);
            }
            if (timerTime != null) {
                params.put("timerTime", timerTime);
            }
            if (extendedCode != null) {
                params.put("extendedCode", extendedCode);
            }
        } catch (UnsupportedEncodingException e) {
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "发送失败！");
            logger.error(e.getMessage());
        }

        HttpRequestParams<Map<String, String>> requestparams = new HttpRequestParams<Map<String, String>>();
        requestparams.setCharSet("UTF-8");
        requestparams.setMethod("POST");
        requestparams.setParams(params);
        requestparams.setUrl(sendSmsApiUrl);
        HttpRequest<Map<String, String>> request;
        if (sendSmsApiUrl.startsWith("https://")) {
            request = new HttpsRequestKV(requestparams, null);
        } else {
            request = new HttpRequestKV(requestparams);
        }
        HttpClient client = new HttpClient();
        String json = null;
        try {
            String mapst = "";
            for (String key : params.keySet()) {
                String value = params.get(key);
                mapst += key + "=" + value + "&";
            }
            mapst = mapst.substring(0, mapst.length() - 1);
            logger.info("request params: " + mapst);
            HttpResponseString res = client.service(request, new HttpResponseStringPraser());
            if (res == null) {
                logger.error("请求接口异常");
                result.put(Constants.RESP_CODE, transFail);
                result.put("respMsg", "请求接口异常！");
                return result;
            }
            if (res.getResultCode().equals(HttpResultCode.SUCCESS)) {
                if (res.getHttpCode() == 200) {
                    json = res.getResult();
                    logger.info("response json: " + json);
                } else {
                    logger.info("请求接口异常,请求码:" + res.getHttpCode());
                    result.put(Constants.RESP_CODE, transFail);
                    result.put("respMsg", "请求接口异常,请求码:" + res.getHttpCode());
                    return result;
                }
            } else {
                logger.info("请求接口网络异常:" + res.getResultCode().getCode());
                result.put(Constants.RESP_CODE, transFail);
                result.put("respMsg", "请求接口网络异常:" + res.getResultCode().getCode());
                return result;
            }
        } catch (Exception e) {
            logger.error("解析失败");
            logger.error(e.getMessage());
            result.put(Constants.RESP_CODE, transFail);
            result.put("respMsg", "解析失败:" + e.getMessage());
            return result;
        }

        if (json != null) {
            ResponseData<SmsResponse[]> data = JsonHelper.fromJson(new TypeToken<ResponseData<SmsResponse[]>>() {
            }, json);
            String code = data.getCode();
            if (tranSuccess.equals(code)) {
                result.put(Constants.RESP_CODE, tranSuccess);
                result.put("respMsg", "发送成功！");
                hclcSmsLogService.updateLog(iRequest, log, "2", "S", json);
                for (SmsResponse d : data.getData()) {
                    logger.info("data:" + d.getMobile() + "," + d.getSmsId() + "," + d.getCustomSmsId());
                }
            }
        }
        logger.info("=============end setSms==================");
        return result;
    }
}
