package com.hand.app.esign.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.hand.hap.core.IRequest;
import com.hand.hap.intergration.dto.HapInterfaceHeader;
import com.hand.hap.intergration.service.IHapApiService;
import com.hand.hap.intergration.service.IHapInterfaceHeaderService;
import com.hand.hap.system.dto.ResponseData;
import com.hand.hap.system.service.impl.BaseServiceImpl;
import com.timevale.esign.sdk.tech.bean.AccountProfile;
import com.timevale.esign.sdk.tech.bean.PersonBean;
import com.timevale.esign.sdk.tech.bean.PosBean;
import com.timevale.esign.sdk.tech.bean.result.AddAccountResult;
import com.timevale.esign.sdk.tech.bean.result.AddSealResult;
import com.timevale.esign.sdk.tech.bean.result.FileDigestSignResult;
import com.timevale.esign.sdk.tech.bean.result.GetAccountProfileResult;
import com.timevale.esign.sdk.tech.bean.seal.PersonTemplateType;
import com.timevale.esign.sdk.tech.bean.seal.SealColor;
import com.timevale.esign.sdk.tech.impl.constants.SignType;
import com.hand.app.esign.bean.AttachmentInfo;
import com.hand.app.esign.bean.SignPage;
import com.hand.app.esign.dto.PersonData;
import com.hand.app.esign.dto.SignHistory;
import com.hand.app.esign.dto.SignInfo;
import com.hand.app.esign.mapper.PersonDataMapper;
import com.hand.app.esign.service.ISignHistoryService;
import com.hand.app.esign.service.ISignInfoService;
import com.hand.app.esign.utils.SignUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.CharsetUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLIntegrityConstraintViolationException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created with IntelliJ IDEA.
 * User: yang
 * Date: 2018/1/17
 * Time: 14:10
 */
@Service
public class SignInfoServiceImpl extends BaseServiceImpl<SignInfo> implements ISignInfoService {
    public static final String RESULT_ERROR = "-1";
    public static final String SIGN_TYPE_USER = "USER";
    private static final String SIGN_TYPE_PLATFORM = "PLATFORM";
    public static final PersonTemplateType TEMPLATE_TYPE_DEFAULT = PersonTemplateType.SQUARE;
    public static final SealColor SEAL_COLOR_DEFAULT = SealColor.RED;


    @Value("${tsign.project.id}")
    private String projectId;
    @Value("${tsign.project.secret}")
    private String projectSecret;
    @Value("${tsign.api.url}")
    private String apiUrl;
    private CloseableHttpClient httpClient;
    @Autowired
    IHapInterfaceHeaderService headerService;
    @Autowired
    private PersonDataMapper personDataMapper;
    @Autowired
    private ISignHistoryService historyService;
    private boolean isProjectInit = false;

    @Resource(name = "restBean")
    IHapApiService restService;

    private Logger logger = LoggerFactory.getLogger(getClass());

    public SignInfoServiceImpl() {
        // ensure e-sign SDK will be init only once
        RequestConfig config = RequestConfig.custom().setConnectTimeout(60 * 1000).setSocketTimeout(15000).build();
        httpClient = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
    }

    private void initProject() {
        if (isProjectInit) {
            return;
        }
        int code = SignUtils.initProject(projectId, projectSecret, apiUrl);
        if (code == SignUtils.CODE_SUCCESS) {
            isProjectInit = true;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResponseData pdfSign(IRequest iRequest, SignInfo person) {
        if (!isProjectInit) {
            initProject();
        }
        ResponseData responseData = new ResponseData(false);

        List<SignHistory> histories = new ArrayList<>(10);
        // get accountId
        String accountId = getCustomerAccount(person);
        if (StringUtils.isBlank(accountId)) {
            responseData.setMessage("can not get accountId");
            return responseData;
        }
        person.setAccountId(accountId);
        String sealData = getPersonSealData(person);
        if (StringUtils.isBlank(sealData)) {
            responseData.setMessage("could not get person seal data");
            return responseData;
        }

        // get pdf file
        byte[] bytes;
        try {
            bytes = getPdf(person.getAttachmentInfo());
        } catch (IOException e) {
            responseData.setMessage("can not get file content");
            return responseData;
        }
        String attachmentId = person.getAttachmentInfo().getAttachmentId();
        // user sign file
        bytes = userPosSign(person, responseData, histories, accountId, sealData, bytes, attachmentId);
        bytes = userKeySign(person, responseData, histories, accountId, sealData, bytes, attachmentId);
        if (bytes == null) {
            logger.debug("user sign failed");
            return responseData;
        }
        // platform sign
        bytes = selfPosSign(person, responseData, histories, bytes, attachmentId);
        bytes = selfKeySign(person, responseData, histories, bytes, attachmentId);
        if (bytes == null) {
            logger.debug("platform sign failed");
            return responseData;
        }
        //upload file
        String uploadStatus = uploadFile(bytes, person.getAttachmentInfo());
        if (uploadStatus == null) {
            responseData.setMessage("upload file failed");
        } else {
            responseData.setSuccess(true);
            responseData.setMessage(uploadStatus);
        }

        if (CollectionUtils.isNotEmpty(histories)) {
            historyService.batchUpdate(iRequest, histories);
        }
        return responseData;
    }


    private byte[] selfKeySign(SignInfo person, ResponseData responseData, List<SignHistory> histories, byte[] bytes, String attachmentId) {
        List<String> selfKeys = person.getSelfKeys();
        if (CollectionUtils.isNotEmpty(selfKeys)) {
            for (String key : selfKeys) {
                if (StringUtils.isBlank(key)) {
                    logger.debug("platform key is empty, skip this");
                    continue;
                }
                PosBean platformKeywordPosBean = SignUtils.getPlatformKeywordPosBean(key);
                FileDigestSignResult customerResult;
                try {
                    customerResult = SignUtils.platformSignByStreamm(bytes, platformKeywordPosBean, SignType.Key);
                } catch (Exception e) {
                    responseData.setMessage("platform key sign failed" + e.getMessage());
                    return null;
                }
                if (customerResult == null || customerResult.getErrCode() != SignUtils.CODE_SUCCESS) {
                    responseData.setMessage("platform sign failed");
                    return null;
                }
                histories.add(historyLog(SIGN_TYPE_USER, attachmentId, customerResult.getSignDetailUrl(), customerResult.getSignServiceId(), key));
                bytes = customerResult.getStream();
            }
        } else {
            logger.debug("not found selfKeyword, skip platform key sign. signInfo: {}", JSON.toJSONString(person));

        }
        return bytes;
    }

    private byte[] userKeySign(SignInfo person, ResponseData responseData, List<SignHistory> histories, String accountId, String sealData, byte[] bytes, String attachmentId) {
        List<String> userKeys = person.getUserKeys();
        if (CollectionUtils.isNotEmpty(userKeys)) {
            for (String key : userKeys) {
                if (StringUtils.isBlank(key)) {
                    logger.debug("user key is empty, skip this");
                    continue;
                }
                PosBean userKeywordPosBean = SignUtils.getUserKeywordPosBean(key);
                FileDigestSignResult customerResult;
                try {
                    customerResult = SignUtils.customerDigestSign(bytes, accountId, sealData, userKeywordPosBean, SignType.Key);
                } catch (Exception e) {
                    responseData.setMessage("custom key sign failed." + e.getMessage());
                    return null;
                }
                if (customerResult == null || customerResult.getErrCode() != SignUtils.CODE_SUCCESS) {
                    responseData.setMessage("customer sign failed");
                    return null;
                }
                histories.add(historyLog(SIGN_TYPE_USER, attachmentId, customerResult.getSignDetailUrl(), customerResult.getSignServiceId(), key));
                bytes = customerResult.getStream();
            }
        } else {
            logger.debug("not found userKeyword, skip user key sign. signInfo: {}", JSON.toJSONString(person));
        }
        return bytes;
    }

    private byte[] selfPosSign(SignInfo person, ResponseData responseData, List<SignHistory> histories, byte[] bytes, String attachmentId) {
        List<SignPage> selfPages = person.getSelfPages();
        if (CollectionUtils.isNotEmpty(selfPages)) {
            for (SignPage selfPage : selfPages) {
                FileDigestSignResult platformResult;
                try {
                    platformResult = SignUtils.platformSignByStreamm(bytes, selfPage.getPage(), selfPage.getX(), selfPage.getY());
                } catch (Exception e) {
                    responseData.setMessage("platform Pos sign failed" + e.getMessage());
                    return null;
                }
                if (platformResult == null || platformResult.getErrCode() != SignUtils.CODE_SUCCESS) {
                    responseData.setMessage("platform sign failed");
                    return null;
                }
                bytes = platformResult.getStream();
                histories.add(historyLog(SIGN_TYPE_PLATFORM, attachmentId, platformResult.getSignDetailUrl(),
                        platformResult.getSignServiceId(), selfPage.getPage(),
                        selfPage.getX(), selfPage.getY()));
            }
        } else {
            logger.debug("not found selfPages, skip platform pos sign. signInfo: {}", JSON.toJSONString(person));
        }
        return bytes;
    }

    private byte[] userPosSign(SignInfo person, ResponseData responseData, List<SignHistory> histories, String accountId, String sealData, byte[] bytes, String attachmentId) {
        List<SignPage> userPages = person.getUserPages();
        if (CollectionUtils.isNotEmpty(userPages)) {
            for (SignPage userPage : userPages) {
                FileDigestSignResult customerResult;
                try {
                    customerResult = SignUtils.customerDigestSign(bytes,
                            accountId,
                            sealData,
                            userPage.getPage(),
                            userPage.getX(),
                            userPage.getY());
                } catch (Exception e) {
                    responseData.setMessage("customer Pos sign failed" + e.getMessage());
                    return null;
                }
                if (customerResult == null || customerResult.getErrCode() != SignUtils.CODE_SUCCESS) {
                    responseData.setMessage("customer sign failed");
                    return null;
                }

                histories.add(historyLog(SIGN_TYPE_USER, attachmentId, customerResult.getSignDetailUrl(),
                        customerResult.getSignServiceId(), userPage.getPage(),
                        userPage.getX(), userPage.getY()));
                bytes = customerResult.getStream();
            }
        } else

        {
            logger.debug("not found userPages, skip user pos sign. signInfo: {}", JSON.toJSONString(person));
        }
        return bytes;
    }

    private String getPersonSealData(SignInfo person) {
        logger.debug("getPersonSealData::{}", JSON.toJSONString(person));
        if (person == null)
            return null;
        String sealData = person.getSealData();
        if (StringUtils.isBlank(sealData)) {
            logger.debug("not get sealData from param");
            String accountId = person.getAccountId();
            if (StringUtils.isBlank(accountId)) {
                return null;
            }

            PersonData personData = new PersonData();
            personData.setAccountId(accountId);
            PersonData data = personDataMapper.selectByPrimaryKey(personData);
            if (data == null || StringUtils.isBlank(data.getSealData())) {
                // not exist in db, create new one ( or fetch from server?)
                AddSealResult addSealResult = SignUtils.addPersonTemplateSeal(accountId, TEMPLATE_TYPE_DEFAULT, SEAL_COLOR_DEFAULT);
                if (SignUtils.CODE_SUCCESS != addSealResult.getErrCode()) {
                    return null;
                }
                data = personDataMapper.selectByPrimaryKey(personData);
                if (data != null && !StringUtils.isBlank(data.getSealData())) {
                    return data.getSealData();
                }
                personData.setSealData(addSealResult.getSealData());
                personData.setColor(SEAL_COLOR_DEFAULT.name());
                personData.setTemplateType(TEMPLATE_TYPE_DEFAULT.name());
                try{
                if(data == null) {
                    personDataMapper.insertSelective(personData);
                }else{
                    personDataMapper.updateByPrimaryKeySelective(personData);
                }
                }catch (Exception e){
                    if(e instanceof SQLIntegrityConstraintViolationException || e instanceof DuplicateKeyException){
                        logger.error("data insert duplicate");
                    }else {
                        throw e;
                    }
                }
                sealData = personData.getSealData();
            } else {
                sealData = data.getSealData();
            }
        }
        return sealData;
    }

    @Override
    public Map<String, Object> getUserAccountOrRegister(IRequest iRequest, SignInfo info) {
        if (!isProjectInit) {
            initProject();
        }
        String accountId = getCustomerAccount(info);
        Map<String, Object> map = new HashMap<>();
        map.put("success", StringUtils.isBlank(accountId));
        map.put("accountId", accountId);
        return map;
    }

    private SignHistory historyLog(String signType, String attachmentId, String detailUrl, String serviceId, String page, int x, int y) {
        SignHistory history = new SignHistory();
        history.setSignType(signType);
        history.setAttachmentId(attachmentId);
        history.setSignServiceId(serviceId);
        history.setSignDetailUrl(detailUrl);
        history.setSignPage(page);
        history.setX(String.valueOf(x));
        history.setY(String.valueOf(y));
        history.set__status("add");
        return history;
    }

    private SignHistory historyLog(String signType, String attachmentId, String detailUrl, String serviceId, String key) {
        SignHistory history = new SignHistory();
        history.setSignType(signType);
        history.setAttachmentId(attachmentId);
        history.setSignServiceId(serviceId);
        history.setSignDetailUrl(detailUrl);
        history.setKeyword(key);
        history.set__status("add");
        return history;
    }

    /**
     * upload file to svc
     *
     * @return null if upload failed
     */
    private String uploadFile(byte[] bytes, AttachmentInfo info) {
        //todo upload file
        logger.debug("params,bytes:{}, attachmentInfo:{}", bytes, JSON.toJSONString(info));
        if (ArrayUtils.isEmpty(bytes) || info == null || StringUtils.isBlank(info.getPkValue()) || StringUtils.isBlank(info.getSourceType())) {
            return null;
        }
        String apiName = info.getUploadApiName();
        String sysName = info.getUploadSysName();
        HapInterfaceHeader headerAndLineDTO = headerService.getHeaderAndLine(sysName, apiName);
        if (headerAndLineDTO == null) {
            logger.debug("headerAndLineDTO is null， apiName:{}, sysName:{}", apiName, sysName);
            return null;
        }
        String url = headerAndLineDTO.getDomainUrl() + headerAndLineDTO.getIftUrl();
        logger.debug("upload url: {}", url);
        HttpPost httpPost = new HttpPost(url);

        //prepare params
        StringBody pkValue = new StringBody(info.getPkValue(), ContentType.MULTIPART_FORM_DATA);
        StringBody userId = new StringBody(info.getUserId(), ContentType.MULTIPART_FORM_DATA);
        StringBody sourceType = new StringBody(info.getSourceType(), ContentType.MULTIPART_FORM_DATA);

        ContentType pdfType = ContentType.create("application/pdf", Consts.UTF_8);

        try {
            HttpEntity entity = MultipartEntityBuilder.create()
                    .setMode(HttpMultipartMode.BROWSER_COMPATIBLE)
                    .addPart("pkvalue", pkValue)
                    .addPart("user_id", userId)
                    .addPart("source_type", sourceType)
                    .addBinaryBody("file", bytes, pdfType, info.getFileName())
                    .setCharset(CharsetUtils.get("utf-8"))
                    .build();
            httpPost.setEntity(entity);
            CloseableHttpResponse response = httpClient.execute(httpPost);
            InputStream content = response.getEntity().getContent();
            String result = IOUtils.toString(content);
            if (StringUtils.equals(RESULT_ERROR, result)) {
                return null;
            }
            return result;
        } catch (IOException e) {
            logger.error("upload attachment failed, attachmentInfo:{}", JSON.toJSONString(info), e);
        }
        return null;
    }

    private byte[] getPdf(AttachmentInfo info) throws IOException {
        if (info == null || StringUtils.isBlank(info.getAttachmentId())) {
            return null;
        }
        String attachmentId = info.getAttachmentId();
        String apiName = info.getDownloadApiName();
        String sysName = info.getDownloadSysName();

        HapInterfaceHeader headerAndLineDTO = headerService.getHeaderAndLine(sysName, apiName);
        if (headerAndLineDTO == null) {
            return null;
        }
        //prepare url
        String url = headerAndLineDTO.getDomainUrl() + headerAndLineDTO.getIftUrl();
        url = url + (url.indexOf("?") > 0 ? "&attachment_id=" : "?attachment_id=") + attachmentId;

        HttpGet httpGet = new HttpGet(url);
        CloseableHttpResponse response = httpClient.execute(httpGet);
        if (HttpStatus.SC_OK != response.getStatusLine().getStatusCode()) {
            logger.error("request target attachment({}) failed,sysName:{}, apiName:{}", attachmentId, sysName, apiName);
            return null;
        }
        InputStream content = response.getEntity().getContent();
        return IOUtils.toByteArray(content);

    }

    private String getCustomerAccount(SignInfo person) {
        // get customer accountId direct if exist,or create one
        Assert.notNull(person, "customer info can not be null");
        String idNo = person.getIdNo();
        String accountId = person.getAccountId();
        if (StringUtils.isNotBlank(accountId)) {
            return accountId;
        }
        // find in db
        if (StringUtils.isBlank(idNo)) {
            throw new IllegalArgumentException("accountId and idNo can not be null");
        }
        SignInfo signInfo = mapper.selectByPrimaryKey(person);
        if (signInfo == null) {
            // not exists in db, then create one
            PersonBean personBean = signInfo2PersonBean(person);
            if (personBean != null) {
                AddAccountResult result = SignUtils.addPersonAccount(personBean);
                if (result.getErrCode() == SignUtils.CODE_SUCCESS) {
                    accountId = result.getAccountId();
                } else if (result.getErrCode() == SignUtils.CODE_ERROR_ACCOUNT_EXIST) {
                    accountId = fetchExistAccountFromServer(idNo);
                }
                if (StringUtils.isBlank(accountId)) {
                    throw new RuntimeException("get account error");
                }
                person.setAccountId(accountId);
                try {
                    mapper.insertSelective(person);
                }catch (Exception e){
                    if(e instanceof SQLIntegrityConstraintViolationException || e instanceof DuplicateKeyException){
                        // 数据重复 do nothing
                        logger.error("insert duplicate", e);
                    }else {
                        logger.error("insert error", e);
                        throw e;
                    }
                }
                return accountId;
            }

        } else {
            return signInfo.getAccountId();
        }
        return null;
    }

    private String fetchExistAccountFromServer(String idNo) {
        GetAccountProfileResult userAccount = SignUtils.getExistUserAccount(idNo);
        if (SignUtils.CODE_SUCCESS == userAccount.getErrCode()) {
            AccountProfile accountInfo = userAccount.getAccountInfo();
            SignInfo signInfo = new SignInfo();
            signInfo.setAccountId(accountInfo.getAccountUid());
            signInfo.setMobile(accountInfo.getMobile());
            signInfo.setIdNo(accountInfo.getIdNo());
            signInfo.setPersonArea(accountInfo.getIdNoType());
            signInfo.setName(accountInfo.getName());
            mapper.insertSelective(signInfo);
            return signInfo.getAccountId();
        }
        return null;
    }

    private PersonBean signInfo2PersonBean(SignInfo person) {
        if (person == null) {
            return null;
        }
        PersonBean personBean = null;
        try {
            String json = JSON.toJSONString(person);
            personBean = JSON.parseObject(json, PersonBean.class);

        } catch (Exception e) {
            logger.error("trans to PersonBean error", e);
        }
        return personBean;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResponseData pdfSignByFile(IRequest iRequest, SignInfo person) {
        if (!isProjectInit) {
            initProject();
        }
        ResponseData responseData = new ResponseData(false);

        List<SignHistory> histories = new ArrayList<>(10);
        // get accountId
        String accountId = getCustomerAccount(person);
        if (StringUtils.isBlank(accountId)) {
            responseData.setMessage("can not get accountId");
            return responseData;
        }
        person.setAccountId(accountId);
        String sealData = getPersonSealData(person);
        if (StringUtils.isBlank(sealData)) {
            responseData.setMessage("could not get person seal data");
            return responseData;
        }

        FileDigestSignResult customerResult;
        String attachmentId = person.getAttachmentInfo().getAttachmentId();
        // user sign file
        //bytes = userPosSign(person, responseData, histories, accountId, sealData, bytes, attachmentId);
        customerResult = userKeySignByFile(person, responseData, histories, accountId, sealData,attachmentId);
        if (customerResult == null|| customerResult.getErrCode() != SignUtils.CODE_SUCCESS) {
            logger.debug("user sign failed");
            return responseData;
        }

        if(customerResult.getDstFilePath() != null){
            String uploadStatus = saveAttachmentInfo(customerResult.getDstFilePath(), person.getAttachmentInfo());
            if (uploadStatus == null) {
                responseData.setMessage("save attachmentinfo failed");
            } else {
                responseData.setSuccess(true);
                responseData.setMessage(uploadStatus);
            }
        }

        if (CollectionUtils.isNotEmpty(histories)) {
            historyService.batchUpdate(iRequest, histories);
        }
        return responseData;
    }

    private FileDigestSignResult userKeySignByFile(SignInfo info, ResponseData responseData, List<SignHistory> histories, String accountId, String sealData, String attachmentId) {
        String srcPdfFile = info.getAttachmentInfo().getSrcPdfFilePath();
        String signedPdfFile = srcPdfFile.split("\\.")[0] + "-signByUser.pdf";
        String signedPdfFileName = info.getAttachmentInfo().getFileName();
        List<String> userKeys = info.getUserKeys();
        FileDigestSignResult customerResult = new FileDigestSignResult();
        if (CollectionUtils.isNotEmpty(userKeys)) {
            for (String key : userKeys) {
                if (StringUtils.isBlank(key)) {
                    logger.debug("user key is empty, skip this");
                    continue;
                }
                PosBean userKeywordPosBean = SignUtils.getUserKeywordPosBean(key);
                try {
                    if(!StringUtils.isBlank(customerResult.getDstFilePath())){
                        srcPdfFile = customerResult.getDstFilePath();
                        signedPdfFile = srcPdfFile.split("\\.")[0] + "-signByUser.pdf";
                    }
                    customerResult = SignUtils.userOrganizeSignByFile(srcPdfFile, signedPdfFile,signedPdfFileName,accountId, sealData, userKeywordPosBean, SignType.Key);
                } catch (Exception e) {
                    responseData.setMessage("custom key sign failed." + e.getMessage());
                    return null;
                }
                if (customerResult == null || customerResult.getErrCode() != SignUtils.CODE_SUCCESS) {
                    responseData.setMessage("customer sign failed");
                    return null;
                }
                histories.add(historyLog(SIGN_TYPE_USER, attachmentId, customerResult.getSignDetailUrl(), customerResult.getSignServiceId(), key));
            }
        } else {
            logger.debug("not found userKeyword, skip user key sign. signInfo: {}", JSON.toJSONString(info));
        }
        return customerResult;
    }

    /**
     * save attachmentInfo
     *
     * @return null if upload failed
     */
    private String saveAttachmentInfo(String signedPath,AttachmentInfo info) {
        logger.debug("params,bytes:{}, attachmentInfo:{}", signedPath, JSON.toJSONString(info));
        if (StringUtils.isEmpty(signedPath) || info == null || StringUtils.isBlank(info.getPkValue()) || StringUtils.isBlank(info.getSourceType())) {
            return null;
        }
        String apiName = info.getSaveApiName();
        String sysName = info.getSaveSysName();

        HapInterfaceHeader headerAndLineDTO = headerService.getHeaderAndLine(sysName, apiName);
        if (headerAndLineDTO == null) {
            logger.debug("headerAndLineDTO is null， apiName:{}, sysName:{}", apiName, sysName);
            return null;
        }
        String url = headerAndLineDTO.getDomainUrl() + headerAndLineDTO.getIftUrl();
        logger.debug("savaAttachment url: {}", url);
        String fileSize = String.valueOf(getFileSize(signedPath));
        String fileType = "pdf";
        String mimeType = "pdf";
        JSONObject params = new JSONObject();
        params.put("table_pk_value", info.getPkValue());
        params.put("user_id", info.getUserId());
        params.put("table_name", info.getSourceType());
        params.put("file_name", info.getFileName());
        params.put("file_path", signedPath);
        params.put("file_size", fileSize);
        params.put("file_type_code", fileType);
        params.put("mime_type", mimeType);

        try {
            JSONObject result = restService.invoke(headerAndLineDTO,params);

            if (StringUtils.equals("S", result.getString("return_status"))) {
                return result.getString("attachment_id");
            }else{
                logger.error("save attachment failed, message:{}", result.getString("message"));
            }
        } catch (Exception e) {
            logger.error("save attachment failed, attachmentInfo:{}", JSON.toJSONString(info), e);
        }
        return null;
    }

    public long getFileSize(String  filePath) {
        File file = new File(filePath);
        if (file.exists() && file.isFile()) {
            return file.length();
        }
        return 0;
    }


}
