package com.hand.app.esign.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.hand.hap.core.IRequest;
import com.hand.hap.intergration.dto.HapInterfaceHeader;
import com.hand.hap.intergration.service.IHapApiService;
import com.hand.hap.intergration.service.IHapInterfaceHeaderService;
import com.hand.hap.system.dto.ResponseData;
import com.hand.hap.system.service.impl.BaseServiceImpl;
import com.timevale.esign.sdk.tech.bean.AccountProfile;
import com.timevale.esign.sdk.tech.bean.OrganizeBean;
import com.timevale.esign.sdk.tech.bean.PosBean;
import com.timevale.esign.sdk.tech.bean.result.AddAccountResult;
import com.timevale.esign.sdk.tech.bean.result.AddSealResult;
import com.timevale.esign.sdk.tech.bean.result.FileDigestSignResult;
import com.timevale.esign.sdk.tech.bean.result.GetAccountProfileResult;
import com.timevale.esign.sdk.tech.bean.seal.OrganizeTemplateType;
import com.timevale.esign.sdk.tech.bean.seal.SealColor;
import com.timevale.esign.sdk.tech.impl.constants.LicenseQueryType;
import com.timevale.esign.sdk.tech.impl.constants.SignType;
import com.hand.app.esign.bean.AttachmentInfo;
import com.hand.app.esign.dto.OrganSignInfo;
import com.hand.app.esign.dto.PersonData;
import com.hand.app.esign.dto.SignHistory;
import com.hand.app.esign.mapper.PersonDataMapper;
import com.hand.app.esign.service.IOrganSignInfoService;
import com.hand.app.esign.service.ISignHistoryService;
import com.hand.app.esign.utils.SignUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.io.File;
import java.sql.SQLIntegrityConstraintViolationException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created with IntelliJ IDEA.
 * User: yang
 * Date: 2018/1/17
 * Time: 14:10
 */
@Service
public class OrganSignInfoServiceImpl extends BaseServiceImpl<OrganSignInfo> implements IOrganSignInfoService {
    public static final String SIGN_TYPE_USER = "USER";
    private static final String SIGN_TYPE_PLATFORM = "PLATFORM";
    public static final OrganizeTemplateType ORGANIZE_TEMPLATE_TYPE_DEFAULT = OrganizeTemplateType.STAR;
    public static final SealColor SEAL_COLOR_DEFAULT = SealColor.RED;

    @Value("${tsign.project.id}")
    private String projectId;
    @Value("${tsign.project.secret}")
    private String projectSecret;
    @Value("${tsign.api.url}")
    private String apiUrl;
    private CloseableHttpClient httpClient;
    @Autowired
    IHapInterfaceHeaderService headerService;
    @Autowired
    private PersonDataMapper personDataMapper;
    @Autowired
    private ISignHistoryService historyService;
    private boolean isProjectInit = false;

    @Resource(name = "restBean")
    IHapApiService restService;

    private Logger logger = LoggerFactory.getLogger(getClass());

    public OrganSignInfoServiceImpl() {
        // ensure e-sign SDK will be init only once
        RequestConfig config = RequestConfig.custom().setConnectTimeout(60 * 1000).setSocketTimeout(15000).build();
        httpClient = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
    }

    private void initProject() {
        if (isProjectInit) {
            return;
        }
        int code = SignUtils.initProject(projectId, projectSecret, apiUrl);
        if (code == SignUtils.CODE_SUCCESS) {
            isProjectInit = true;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResponseData pdfSignByOrgan(IRequest iRequest, OrganSignInfo info) {
        if (!isProjectInit) {
            initProject();
        }
        ResponseData responseData = new ResponseData(false);

        List<SignHistory> histories = new ArrayList<>(10);
        // get accountId
        String accountId = getOrganAccount(info);
        if (StringUtils.isBlank(accountId)) {
            responseData.setMessage("can not get accountId");
            return responseData;
        }
        info.setAccountId(accountId);
        String sealData = getOrganSealData(info);
        if (StringUtils.isBlank(sealData)) {
            responseData.setMessage("could not get organize seal data");
            return responseData;
        }

        FileDigestSignResult customerResult;
        String attachmentId = info.getAttachmentInfo().getAttachmentId();
        // user sign file
       //bytes = userPosSign(person, responseData, histories, accountId, sealData, bytes, attachmentId);
        customerResult = userOrganKeySign(info, responseData, histories, accountId, sealData,attachmentId);
        if (customerResult == null|| customerResult.getErrCode() != SignUtils.CODE_SUCCESS) {
            logger.debug("user sign failed");
            return responseData;
        }

        if(customerResult.getDstFilePath() != null){
            String uploadStatus = saveAttachmentInfo(customerResult.getDstFilePath(), info.getAttachmentInfo());
            if (uploadStatus == null) {
               responseData.setMessage("save attachmentinfo failed");
            } else {
              responseData.setSuccess(true);
              responseData.setMessage(uploadStatus);
            }
        }

        if (CollectionUtils.isNotEmpty(histories)) {
            historyService.batchUpdate(iRequest, histories);
        }
        return responseData;
    }

    private FileDigestSignResult userOrganKeySign(OrganSignInfo info, ResponseData responseData, List<SignHistory> histories, String accountId, String sealData, String attachmentId) {
        String srcPdfFile = info.getAttachmentInfo().getSrcPdfFilePath();
        String signedPdfFile = srcPdfFile.split("\\.")[0] + "-signByUser.pdf";
        String signedPdfFileName = info.getAttachmentInfo().getFileName();
        List<String> userKeys = info.getUserKeys();
        FileDigestSignResult customerResult = new FileDigestSignResult();
        if (CollectionUtils.isNotEmpty(userKeys)) {
            for (String key : userKeys) {
                if (StringUtils.isBlank(key)) {
                    logger.debug("user key is empty, skip this");
                    continue;
                }
                PosBean OrganizeKeywordPosBean = SignUtils.getOrganizeKeywordPosBean(key);
                try {
                    if(!StringUtils.isBlank(customerResult.getDstFilePath())){
                        srcPdfFile = customerResult.getDstFilePath();
                        signedPdfFile = srcPdfFile.split("\\.")[0] + "-signByUser.pdf";
                    }
                    customerResult = SignUtils.userOrganizeSignByFile(srcPdfFile, signedPdfFile,signedPdfFileName,accountId, sealData, OrganizeKeywordPosBean, SignType.Key);
                } catch (Exception e) {
                    responseData.setMessage("custom key sign failed." + e.getMessage());
                    return null;
                }
                if (customerResult == null || customerResult.getErrCode() != SignUtils.CODE_SUCCESS) {
                    responseData.setMessage("customer sign failed");
                    return null;
                }
                histories.add(historyLog(SIGN_TYPE_USER, attachmentId, customerResult.getSignDetailUrl(), customerResult.getSignServiceId(), key));
            }
        } else {
            logger.debug("not found userKeyword, skip user key sign. signInfo: {}", JSON.toJSONString(info));
        }
        return customerResult;
    }

    private SignHistory historyLog(String signType, String attachmentId, String detailUrl, String serviceId, String key) {
        SignHistory history = new SignHistory();
        history.setSignType(signType);
        history.setAttachmentId(attachmentId);
        history.setSignServiceId(serviceId);
        history.setSignDetailUrl(detailUrl);
        history.setKeyword(key);
        history.set__status("add");
        return history;
    }

    private String getOrganSealData(OrganSignInfo info) {
        logger.debug("getOrganSealData::{}", JSON.toJSONString(info));
        if (info == null)
            return null;
        String hText = info.gethText();
        String qText = info.getqText();
        String sealData = info.getSealData();
        if (StringUtils.isBlank(sealData)) {
            logger.debug("not get sealData from param");
            String accountId = info.getAccountId();
            if (StringUtils.isBlank(accountId)) {
                return null;
            }
            PersonData personData = new PersonData();
            personData.setAccountId(accountId);
            PersonData data = personDataMapper.selectByPrimaryKey(personData);
            if (data == null || StringUtils.isBlank(data.getSealData())) {
                // not exist in db, create new one ( or fetch from server?)
                AddSealResult addSealResult = SignUtils.addOrganizeTemplateSeal(accountId,ORGANIZE_TEMPLATE_TYPE_DEFAULT, SEAL_COLOR_DEFAULT, hText, qText);
                if (SignUtils.CODE_SUCCESS != addSealResult.getErrCode()) {
                    return null;
                }
                data = personDataMapper.selectByPrimaryKey(personData);
                if (data != null && !StringUtils.isBlank(data.getSealData())) {
                    return data.getSealData();
                }
                personData.setSealData(addSealResult.getSealData());
                personData.setColor(SEAL_COLOR_DEFAULT.name());
                personData.setTemplateType(ORGANIZE_TEMPLATE_TYPE_DEFAULT.name());
                personData.setHtext(hText);
                personData.setQtext(qText);
                try{
                    if(data == null) {
                        personDataMapper.insertSelective(personData);
                    }else{
                        personDataMapper.updateByPrimaryKeySelective(personData);
                    }
                }catch (Exception e){
                    if(e instanceof SQLIntegrityConstraintViolationException || e instanceof DuplicateKeyException){
                        logger.error("data insert duplicate");
                    }else {
                        throw e;
                    }
                }
                sealData = personData.getSealData();
            } else {
                sealData = data.getSealData();
            }
        }
        return sealData;
    }

    @Override
    public Map<String, Object> getUserAccountOrRegisterByOrgan(IRequest iRequest, OrganSignInfo info) {
        if (!isProjectInit) {
            initProject();
        }
        String accountId = getOrganAccount(info);
        Map<String, Object> map = new HashMap<>();
        map.put("success", StringUtils.isBlank(accountId));
        map.put("accountId", accountId);
        return map;
    }

    private String getOrganAccount(OrganSignInfo organ) {
        // get customer accountId direct if exist,or create one
        Assert.notNull(organ, "organize info can not be null");
        String organCode = organ.getOrganCode();
        String accountId = organ.getAccountId();
        if (StringUtils.isNotBlank(accountId)) {
            return accountId;
        }
        // find in db
        if (StringUtils.isBlank(organCode)) {
            throw new IllegalArgumentException("accountId and idNo can not be null");
        }
        OrganSignInfo signInfo = mapper.selectByPrimaryKey(organ);
        if (signInfo == null) {
            // not exists in db, then create one
            OrganizeBean organizeBean = signInfo2OrganizeBean(organ);
            if (organizeBean != null) {
                AddAccountResult result = SignUtils.addOrganizeAccount(organizeBean);
                if (result.getErrCode() == SignUtils.CODE_SUCCESS) {
                    accountId = result.getAccountId();
               }
               else if (result.getErrCode() == SignUtils.CODE_ERROR_ACCOUNT_EXIST) {
                      accountId = fetchExistAccountFromServer(organCode,LicenseQueryType.MERGE);
            }
                if (StringUtils.isBlank(accountId)) {
                    throw new RuntimeException("get account error");
                }
                    organ.setAccountId(accountId);
                try {
                    mapper.insertSelective(organ);
                }catch (Exception e){
                    if(e instanceof SQLIntegrityConstraintViolationException || e instanceof DuplicateKeyException){
                        // 数据重复 do nothing
                        logger.error("insert duplicate", e);
                    }else {
                        logger.error("insert error", e);
                        throw e;
                    }
                }
                return accountId;
            }

        } else {
            return signInfo.getAccountId();
        }
        return null;
    }

    private String fetchExistAccountFromServer(String idNo , LicenseQueryType area) {
        GetAccountProfileResult userAccount = SignUtils.getExistUserAccount(idNo,area);
        if (SignUtils.CODE_SUCCESS == userAccount.getErrCode()) {
            AccountProfile accountInfo = userAccount.getAccountInfo();
            OrganSignInfo signInfo = new OrganSignInfo();
            signInfo.setAccountId(accountInfo.getAccountUid());
            signInfo.setOrganCode(accountInfo.getIdNo());
            signInfo.setName(accountInfo.getName());
            mapper.insertSelective(signInfo);
            return signInfo.getAccountId();
        }
        return null;
    }


    private OrganizeBean signInfo2OrganizeBean(OrganSignInfo organ) {
        if (organ == null) {
            return null;
        }
        OrganizeBean organizeBean = null;
        try {
            String json = JSON.toJSONString(organ);
            organizeBean = JSON.parseObject(json, OrganizeBean.class);

        } catch (Exception e) {
            logger.error("trans to PersonBean error", e);
        }
        return organizeBean;
    }

    /**
     * save attachmentInfo
     *
     * @return null if upload failed
     */
    private String saveAttachmentInfo(String signedPath,AttachmentInfo info) {
        logger.debug("params,bytes:{}, attachmentInfo:{}", signedPath, JSON.toJSONString(info));
        if (StringUtils.isEmpty(signedPath) || info == null || StringUtils.isBlank(info.getPkValue()) || StringUtils.isBlank(info.getSourceType())) {
            return null;
        }
        String apiName = info.getSaveApiName();
        String sysName = info.getSaveSysName();

        HapInterfaceHeader hapInterfaceHeader = this.headerService.getHeaderAndLine(sysName, apiName);
        if (hapInterfaceHeader == null) {
            logger.debug("hapInterfaceHeader is null， apiName:{}, sysName:{}", apiName, sysName);
            return null;
        }
        String url = hapInterfaceHeader.getDomainUrl() + hapInterfaceHeader.getIftUrl();
        logger.debug("savaAttachment url: {}", url);

        String fileSize = String.valueOf(getFileSize(signedPath));
        String fileType = "pdf";
        String mimeType = "pdf";

        JSONObject params = new JSONObject();
        params.put("table_pk_value", info.getPkValue());
        params.put("user_id", info.getUserId());
        params.put("table_name", info.getSourceType());
        params.put("file_name", info.getFileName());
        params.put("file_path", signedPath);
        params.put("file_size", fileSize);
        params.put("file_type_code", fileType);
        params.put("mime_type", mimeType);

        try {
            JSONObject result = restService.invoke(hapInterfaceHeader,params);

            if (StringUtils.equals("S", result.getString("return_status"))) {
                return result.getString("attachment_id");
            }else{
                logger.error("save attachment failed, message:{}", result.getString("message"));
            }
        } catch (Exception e) {
            logger.error("save attachment failed, attachmentInfo:{}", JSON.toJSONString(info), e);
        }
        return null;
    }

    public long getFileSize(String  filePath) {
        File file = new File(filePath);
        if (file.exists() && file.isFile()) {
            return file.length();
        }
        return 0;
    }
}
