package hls.support.core.wechat.utils;

import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.api.impl.WxCpServiceImpl;
import me.chanjar.weixin.cp.config.WxCpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.api.impl.WxMpServiceImpl;
import net.coobird.thumbnailator.Thumbnails;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.util.Formatter;

/**
 * Created by weck on 12/01/2018 1:37 PM
 */
public class WeChatUtils {

    //2MB
    private static final Integer standardSize = 2 * 1048576;
    private static Logger log = LoggerFactory.getLogger(WeChatUtils.class);

    /**
     * 初始化企业号sdk的service
     *
     * @param corpId 企业号的corpId
     * @param secret 企业号或者企业应用的secret
     * @return WxCpService对象
     */
    public static WxCpService initWxCpService(String corpId, String secret, Integer agentId, String proxyHost, Integer proxyPort) {
        WxCpInMemoryConfigStorage config = new WxCpInMemoryConfigStorage();
        config.setCorpId(corpId);
        config.setCorpSecret(secret);
        if (proxyHost != null && proxyPort != null) {
            config.setHttpProxyHost(proxyHost);
            config.setHttpProxyPort(proxyPort);
        }
        if (agentId != null) {
            config.setAgentId(agentId);
        }
        WxCpService wxCpService = new WxCpServiceImpl();
        wxCpService.setWxCpConfigStorage(config);
        return wxCpService;
    }

    /**
     * 初始化非企业号sdk的service
     *
     * @param appId  非企业号的appId
     * @param secret 非企业号的secret
     * @return WxMpService对象
     */
    public static WxMpService initWxMpService(String appId, String secret, String proxyHost, Integer proxyPort) {
        WxMpInMemoryConfigStorage config = new WxMpInMemoryConfigStorage();
        config.setAppId(appId);
        config.setSecret(secret);
        if (proxyHost != null && proxyPort != null) {
            config.setHttpProxyHost(proxyHost);
            config.setHttpProxyPort(proxyPort);
        }
        WxMpService wxMpService = new WxMpServiceImpl();
        wxMpService.setWxMpConfigStorage(config);
        return wxMpService;
    }

    /**
     * 压缩图片
     *
     * @param image 前端传入图片
     * @param dest  目标文件
     * @return boolean
     */
    public static boolean compressImage(MultipartFile image, File dest) {
        if (image.getSize() > standardSize) {
            try {
                Thumbnails.of(image.getInputStream()).scale((float) standardSize / image.getSize()).toFile(dest);
            } catch (IOException e) {
                log.error("compress image failed:", e);
                return false;
            }
        } else {
            try {
                image.transferTo(dest);
            } catch (IOException e) {
                log.error("save image failed:", e);
                return false;
            }
        }
        return true;
    }

    /**
     * 将id字符串转换成固定格式
     *
     * @param ids 多个id用,分隔的字符串
     * @return string
     */
    public static String formatIds(String ids, Boolean isCorp) {
        ids = StringUtils.trim(ids);
        if (StringUtils.isNotEmpty(ids)) {
            if (StringUtils.endsWith(ids, ","))
                ids = StringUtils.removeEnd(ids, ",");
            if (isCorp)
                StringUtils.replace(ids, ",", "|");
        }
        return ids;
    }

    public static String[] formatIds(String ids) {
        ids = StringUtils.trim(ids);
        if (StringUtils.isNotEmpty(ids)) {
            if (StringUtils.endsWith(ids, ","))
                ids = StringUtils.removeEnd(ids, ",");
            return StringUtils.split(ids, ",");
        }
        return null;
    }

    public static String byteToHex(final byte[] hash) {
        Formatter formatter = new Formatter();
        for (byte b : hash) {
            formatter.format("%02x", b);
        }
        String result = formatter.toString();
        formatter.close();
        return result;

    }
}
