package hls.support.core.wechat.service.impl;

import com.hand.hap.core.IRequest;
import com.hand.hap.system.dto.ResponseData;
import hls.support.core.wechat.dto.MPNewsThumbPicture;
import hls.support.core.wechat.dto.WechatAccount;
import hls.support.core.wechat.dto.WechatEnterpriseApp;
import hls.support.core.wechat.formbean.Article;
import hls.support.core.wechat.formbean.WxMessageBean;
import hls.support.core.wechat.formbean.TemplateData;
import hls.support.core.wechat.formbean.TemplateMessage;
import hls.support.core.wechat.mapper.MPNewsThumbPictureMapper;
import hls.support.core.wechat.mapper.WechatAccountMapper;
import hls.support.core.wechat.mapper.WechatEnterpriseAppMapper;
import hls.support.core.wechat.service.IWechatMessageService;
import hls.support.core.wechat.utils.WeChatUtils;
import me.chanjar.weixin.common.api.WxConsts;
import me.chanjar.weixin.common.bean.result.WxMediaUploadResult;
import me.chanjar.weixin.common.exception.WxErrorException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.bean.WxCpMessage;
import me.chanjar.weixin.cp.bean.article.NewArticle;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.bean.WxMpMassNews;
import me.chanjar.weixin.mp.bean.WxMpMassOpenIdsMessage;
import me.chanjar.weixin.mp.bean.WxMpMassTagMessage;
import me.chanjar.weixin.mp.bean.kefu.WxMpKefuMessage;
import me.chanjar.weixin.mp.bean.material.WxMpMaterial;
import me.chanjar.weixin.mp.bean.material.WxMpMaterialUploadResult;
import me.chanjar.weixin.mp.bean.result.WxMpMassUploadResult;
import me.chanjar.weixin.mp.bean.template.WxMpTemplate;
import me.chanjar.weixin.mp.bean.template.WxMpTemplateData;
import me.chanjar.weixin.mp.bean.template.WxMpTemplateIndustry;
import me.chanjar.weixin.mp.bean.template.WxMpTemplateMessage;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

import static hls.support.core.wechat.formbean.WxMessageBean.*;
import static hls.support.core.wechat.utils.WeChatUtils.formatIds;

/**
 * Created by weck on 06/01/2018 10:20 AM
 */

@Service
@Transactional(rollbackFor = Exception.class)
public class WechatMessageServiceImpl implements IWechatMessageService {

    private static Logger logger = LoggerFactory.getLogger(WechatMessageServiceImpl.class);
    private Integer INVALID_DATA = -1;
    private Integer CORP = 1;
    private Integer SUBS = 2;
    private Integer SERV = 3;
    @Autowired
    private WechatEnterpriseAppMapper wechatEnterpriseAppMapper;
    @Autowired
    private WechatAccountMapper wechatAccountMapper;
    @Autowired
    private MPNewsThumbPictureMapper mpNewsThumbPictureMapper;
    @Value("#{configProperties['file.upload.dir']}")
    private String uploadDir;
    @Value("#{configProperties['ftp.upload.dir']}")
    private String ftpUploadDir;

    @Value("#{configProperties['http.proxy.port']}")
    private Integer proxyPort;

    @Value("#{configProperties['http.proxy.host']}")
    private String proxyHost;

    @Override
    public ResponseData getAllTemplate(IRequest requestCtx, TemplateMessage templateMessage) {
        ResponseData responseData = new ResponseData(false);
        Integer type = judgeType(templateMessage.getAccountId(), templateMessage.getWechatAppId(), templateMessage.getWxCode());
        if (!type.equals(SERV)) {
            responseData.setMessage("invalid account type");
            return responseData;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(templateMessage.getAccountId());
        if (account == null) {
            responseData.setMessage("target account could not found.");
            return responseData;
        }
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        try {
            List<WxMpTemplate> templates = wxMpService.getTemplateMsgService().getAllPrivateTemplate();
            responseData.setRows(templates);
            responseData.setSuccess(true);
            return responseData;
        } catch (WxErrorException e) {
            logger.error("get template list failed.", e);
            responseData.setMessage("get template list failed," + e.getMessage());
            return responseData;
        }
    }

    @Override
    public ResponseData delTemplate(IRequest requestCtx, TemplateMessage templateMessage) {
        ResponseData responseData = new ResponseData(false);
        Integer type = judgeType(templateMessage.getAccountId(), templateMessage.getWechatAppId(), templateMessage.getWxCode());
        if (!type.equals(SERV)) {
            responseData.setMessage("invalid account type");
            return responseData;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(templateMessage.getAccountId());
        if (account == null) {
            responseData.setMessage("target account could not found.");
            return responseData;
        }
        if (StringUtils.isEmpty(templateMessage.getTemplateId())) {
            responseData.setMessage("templateId is empty");
            return responseData;
        }
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        try {
            wxMpService.getTemplateMsgService().delPrivateTemplate(templateMessage.getTemplateId());
        } catch (WxErrorException e) {
            logger.error("delete template failed. ", e);
            responseData.setMessage("delete template failed." + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    @Override
    public ResponseData getIndustry(IRequest requestCtx, TemplateMessage templateMessage) {
        ResponseData responseData = new ResponseData(false);
        Integer type = judgeType(templateMessage.getAccountId(), templateMessage.getWechatAppId(), templateMessage.getWxCode());
        if (!type.equals(SERV)) {
            responseData.setMessage("invalid account type");
            return responseData;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(templateMessage.getAccountId());
        if (account == null) {
            responseData.setMessage("target account could not found.");
            return responseData;
        }
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        try {
            WxMpTemplateIndustry industry = wxMpService.getTemplateMsgService().getIndustry();
            List<Object> objects = new ArrayList<>();
            objects.add(industry);
            responseData.setRows(objects);
            responseData.setSuccess(true);
            return responseData;
        } catch (WxErrorException e) {
            logger.error("get industry failed,", e);
            responseData.setMessage("get industry failed," + e.getMessage());
            return responseData;
        }
    }

    @Override
    public ResponseData setIndustry(IRequest requestCtx, TemplateMessage templateMessage) {
        ResponseData responseData = new ResponseData(false);
        Integer type = judgeType(templateMessage.getAccountId(), templateMessage.getWechatAppId(), templateMessage.getWxCode());
        if (templateMessage.getPrimaryIndustry() == null) {
            responseData.setMessage("industry is empty.");
        }
        if (!type.equals(SERV)) {
            responseData.setMessage("invalid account type");
            return responseData;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(templateMessage.getAccountId());
        if (account == null) {
            responseData.setMessage("target account could not found.");
            return responseData;
        }
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        WxMpTemplateIndustry wxMpTemplateIndustry = new WxMpTemplateIndustry();
        WxMpTemplateIndustry.Industry primaryIndustry = new WxMpTemplateIndustry.Industry();
        WxMpTemplateIndustry.Industry secondIndustry = new WxMpTemplateIndustry.Industry();
        BeanUtils.copyProperties(templateMessage.getPrimaryIndustry(), primaryIndustry);
        BeanUtils.copyProperties(templateMessage.getSecondIndustry(), secondIndustry);
        wxMpTemplateIndustry.setPrimaryIndustry(primaryIndustry);
        wxMpTemplateIndustry.setSecondIndustry(secondIndustry);
        try {
            wxMpService.getTemplateMsgService().setIndustry(wxMpTemplateIndustry);
        } catch (WxErrorException e) {
            logger.error("set industry failed.", e);
            responseData.setMessage("set industry failed: " + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    @Override
    public ResponseData addTemplate(IRequest requestCtx, TemplateMessage templateMessage) {
        ResponseData responseData = new ResponseData(false);
        Integer type = judgeType(templateMessage.getAccountId(), templateMessage.getWechatAppId(), templateMessage.getWxCode());
        if (StringUtils.isEmpty(templateMessage.getShortTemplateId())) {
            responseData.setMessage("shortTemplateId is empty.");
        }
        if (!type.equals(SERV)) {
            responseData.setMessage("invalid account type");
            return responseData;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(templateMessage.getAccountId());
        if (account == null) {
            responseData.setMessage("target account could not found.");
            return responseData;
        }
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        try {
            String templateId = wxMpService.getTemplateMsgService().addTemplate(templateMessage.getShortTemplateId());
        } catch (WxErrorException e) {
            logger.error("add template failed.", e);
            responseData.setMessage("add template failed: " + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }


    @Override
    public ResponseData sendTemplateMsg(IRequest requestCtx, TemplateMessage templateMessage) {
        ResponseData responseData = new ResponseData(false);
        Integer type = judgeType(templateMessage.getAccountId(), templateMessage.getWechatAppId(), templateMessage.getWxCode());
        if (StringUtils.isEmpty(templateMessage.getShortTemplateId())) {
            responseData.setMessage("shortTemplateId is empty.");
        }
        if (!type.equals(SERV)) {
            responseData.setMessage("invalid account type");
            return responseData;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(templateMessage.getAccountId());
        if (account == null) {
            responseData.setMessage("target account could not found.");
            return responseData;
        }
        if (CollectionUtils.isEmpty(templateMessage.getTemplateDatas())
                || StringUtils.isEmpty(templateMessage.getToUser())
                || StringUtils.isEmpty(templateMessage.getShortTemplateId())) {
            responseData.setMessage("message data is empty or templateId is empty or target user is empty");
            return responseData;
        }
        List<WxMpTemplateData> dataList = new ArrayList<>();
        for (TemplateData data : templateMessage.getTemplateDatas()) {
            WxMpTemplateData templateData = new WxMpTemplateData();
            BeanUtils.copyProperties(data, templateData);
            dataList.add(templateData);
        }
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        WxMpTemplateMessage message = WxMpTemplateMessage.builder()
                .templateId(templateMessage.getTemplateId())
                .data(dataList)
                .url(templateMessage.getUrl())
                .toUser(templateMessage.getToUser())
                .build();
        try {
            wxMpService.getTemplateMsgService().sendTemplateMsg(message);
        } catch (WxErrorException e) {
            logger.error("send template message error.", e);
            responseData.setMessage("send template message error," + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    /**
     * 发送客服消息
     */
    @Override
    public ResponseData sendKefuMessage(IRequest requestCtx, WxMessageBean messageBean) {
        switch (messageBean.getMessageType()) {
            case MESSAGE_TEXT:
                return sendKefuText(messageBean);
            case MESSAGE_IMAGE:
                return sendKefuImage(messageBean);
            case MESSAGE_MPNEWS:
                return sendKefuMpNews(messageBean);
            case MESSAGE_VIDEO:
                //todo
                return new ResponseData();
            case MESSAGE_VOICE:
                //todo
                return new ResponseData();
            default:
                logger.error("Invalid message type.");
                ResponseData responseData = new ResponseData(false);
                responseData.setMessage("Invalid message type");
                return responseData;
        }
    }

    /**
     * 发送客服图文消息
     *
     * @param messageBean 消息对象
     * @return responsedata
     */
    private ResponseData sendKefuMpNews(WxMessageBean messageBean) {
        ResponseData responseData = new ResponseData(false);
        Integer type = judgeType(messageBean.getAccountId(), messageBean.getAppId(), messageBean.getWxCode());
        if (type.equals(INVALID_DATA) ||
                StringUtils.isEmpty(messageBean.getOpenId()) && StringUtils.isEmpty(messageBean.getUserIds())) {
            responseData.setMessage("parameter data is invalid");
            return responseData;
        }
        if (type.equals(CORP)) {
            return sendCpMPNews(messageBean);
        } else {
            WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
            WechatAccount account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
            WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
            List<WxMpKefuMessage.WxArticle> articles = new ArrayList<>();
            for (Article article : messageBean.getArticles()) {
                WxMpKefuMessage.WxArticle wxArticle = new WxMpKefuMessage.WxArticle();
                wxArticle.setTitle(article.getTitle());
                wxArticle.setUrl(article.getUrl());
                wxArticle.setPicUrl(article.getPicUrl());
                wxArticle.setDescription(article.getDescription());
                articles.add(wxArticle);
            }
            WxMpKefuMessage message = WxMpKefuMessage.NEWS().articles(articles).toUser(messageBean.getOpenId()).build();
            try {
                wxMpService.getKefuService().sendKefuMessage(message);
            } catch (WxErrorException e) {
                logger.error("send kefu news message failed: ", e);
                responseData.setMessage("send kefu news message failed.");
                return responseData;
            }
            responseData.setSuccess(true);
            return responseData;
        }
    }

    /**
     * 发送客服图片消息
     *
     * @param messageBean 消息bean
     * @return responsedata
     */
    private ResponseData sendKefuImage(WxMessageBean messageBean) {
        ResponseData responseData = new ResponseData(false);
        Integer type = judgeType(messageBean.getAccountId(), messageBean.getAppId(), messageBean.getWxCode());
        if (type.equals(INVALID_DATA) ||
                StringUtils.isEmpty(messageBean.getOpenId()) && StringUtils.isEmpty(messageBean.getUserIds())) {
            responseData.setMessage("parameter data is invalid");
            return responseData;
        }
        if (type.equals(CORP)) {
            return sendCpImage(messageBean);
        } else {
            WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
            WechatAccount account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
            WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
            try {
                WxMediaUploadResult uploadMediaRes = wxMpService.getMaterialService()
                        .mediaUpload(WxConsts.MassMsgType.IMAGE, FilenameUtils.getExtension(messageBean.getFileName()),
                                new FileInputStream(new File(uploadDir + messageBean.getFileName())));
                WxMpKefuMessage message = WxMpKefuMessage.IMAGE().mediaId(uploadMediaRes.getMediaId())
                        .toUser(messageBean.getOpenId()).build();
                wxMpService.getKefuService().sendKefuMessage(message);
            } catch (WxErrorException | FileNotFoundException e) {
                logger.error("send kefu image message failed, ", e);
                responseData.setMessage("send kefu image message failed");
                return responseData;
            }
            responseData.setSuccess(true);
            return responseData;
        }
    }

    /**
     * 发送客服文本消息
     *
     * @param messageBean 消息bean
     * @return responsedata
     */
    private ResponseData sendKefuText(WxMessageBean messageBean) {
        Integer type = judgeType(messageBean.getAccountId(), messageBean.getAppId(), messageBean.getWxCode());
        if (type.equals(INVALID_DATA) ||
                StringUtils.isEmpty(messageBean.getOpenId()) && !type.equals(CORP)) {
            ResponseData responseData = new ResponseData(false);
            responseData.setMessage("parameter data is invalid");
            return responseData;
        }
        if (type.equals(CORP)) {
            return sendCpText(messageBean);
        } else {
            ResponseData responseData = new ResponseData(false);
            WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
            WechatAccount account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
            WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
            WxMpKefuMessage message = WxMpKefuMessage
                    .TEXT().toUser(messageBean.getOpenId()).content(messageBean.getText()).build();
            try {
                wxMpService.getKefuService().sendKefuMessage(message);
            } catch (WxErrorException e) {
                logger.error("send kefu text message failed", e);
                responseData.setMessage("send kefu text message failed.");
                return responseData;
            }
            responseData.setSuccess(true);
            return responseData;
        }
    }

    /**
     * 发送模板文本消息
     *
     * @param messageBean 消息bean
     * @return responsedata
     */
    public ResponseData sendTemplateText(WxMessageBean messageBean) {
        System.out.println("****************AccountId:"+messageBean.getAccountId());
        System.out.println("****************AppId:"+messageBean.getAppId());
        System.out.println("****************WxCode:"+messageBean.getWxCode());
        Integer type = judgeType(messageBean.getAccountId(), messageBean.getAppId(), messageBean.getWxCode());
        System.out.println("****************type:"+type);
        System.out.println("****************equals:"+type.equals(INVALID_DATA)+"---"+StringUtils.isEmpty(messageBean.getOpenId())+"---"+!type.equals(CORP));
        if (type.equals(INVALID_DATA) ||
                StringUtils.isEmpty(messageBean.getOpenId()) && !type.equals(CORP)) {
            ResponseData responseData = new ResponseData(false);
            responseData.setMessage("parameter data is invalid");
            return responseData;
        }
        if (type.equals(CORP)) {
            return sendCpText(messageBean);
        } else if (type.equals(SERV)) {
            ResponseData responseData = new ResponseData(false);
            WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
            WechatAccount account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
            WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
            WxMpTemplateMessage templateMessage = WxMpTemplateMessage.builder()
                    .toUser(messageBean.getOpenId())
                    .templateId(messageBean.getTemplateId())
                    .url(messageBean.getUrl())
                    .build();
            try {
                wxMpService.getTemplateMsgService().sendTemplateMsg(templateMessage);
                responseData.setSuccess(true);
            } catch (WxErrorException e) {
                logger.error("send template message failed{}", e);
                responseData.setMessage("send template message failed." + e.getMessage());
                return responseData;
            }
            responseData.setSuccess(true);
            return responseData;
        } else {
            ResponseData responseData = new ResponseData(false);
            responseData.setMessage("选择的微信应用不是服务号应用。");
            return responseData;
        }
    }

    /**
     * 企业号发送图片消息
     *
     * @param messageBean 消息bean
     * @return bool
     */
    private ResponseData sendCpImage(WxMessageBean messageBean) {
        ResponseData responseData = new ResponseData(false);
        WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
        Integer agentId = Integer.valueOf(app.getAgentId());
        WxCpService wxCpService = WeChatUtils.initWxCpService(app.getCorpId(), app.getSecret(), Integer.valueOf(app.getAgentId()), proxyHost, proxyPort);
        String partyIds = formatIds(messageBean.getPartyIds(), true);
        String tagIds = formatIds(messageBean.getTagIds(), true);
        String userIds = formatIds(messageBean.getUserIds(), true);
        try {
            WxMediaUploadResult res = wxCpService.getMediaService()
                    .upload("image", new File(uploadDir + messageBean.getFileName()));
            WxCpMessage message;
            if (StringUtils.isEmpty(partyIds) && StringUtils.isEmpty(tagIds) && StringUtils.isEmpty(userIds)) {
                message = WxCpMessage.IMAGE()
                        .mediaId(res.getMediaId())
                        .agentId(agentId)
                        .toUser("@all")
                        .build();
            } else {
                message = WxCpMessage.IMAGE()
                        .mediaId(res.getMediaId())
                        .agentId(agentId)
                        .toParty(partyIds)
                        .toTag(tagIds)
                        .toUser(userIds)
                        .build();
            }
            wxCpService.messageSend(message);
        } catch (WxErrorException e) {
            logger.error("upload media file failed or send image message failed：", e);
            responseData.setMessage("upload media file failed or send image message failed：" + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    /**
     * 非企业好发送图片消息
     *
     * @param messageBean 消息bean
     * @return bool
     */
    private ResponseData sendMpImage(WxMessageBean messageBean) {
        ResponseData responseData = new ResponseData(false);
        WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        String[] openIds = formatIds(messageBean.getOpenIds());
        String tagId = messageBean.getTagId();
        try {
            WxMpMaterial material = new WxMpMaterial();
            material.setFile(new File(uploadDir + messageBean.getFileName()));
            material.setName(messageBean.getFileName());
            WxMpMaterialUploadResult uploadMaterialRes = wxMpService.getMaterialService()
                    .materialFileUpload(WxConsts.MaterialType.IMAGE, material);
            if (ArrayUtils.isNotEmpty(openIds)) {
                WxMpMassOpenIdsMessage massMessage = new WxMpMassOpenIdsMessage();
                massMessage.setMsgType(WxConsts.MassMsgType.IMAGE);
                massMessage.setMediaId(uploadMaterialRes.getMediaId());
                for (String openId : openIds) {
                    if (StringUtils.isNotEmpty(openId))
                        massMessage.getToUsers().add(openId);
                }
                wxMpService.getMassMessageService().massOpenIdsMessageSend(massMessage);
            } else {
                WxMpMassTagMessage tagMessage = new WxMpMassTagMessage();
                tagMessage.setMsgType(WxConsts.MassMsgType.IMAGE);
                tagMessage.setMediaId(uploadMaterialRes.getMediaId());
                if (StringUtils.isNotEmpty(tagId)) {
                    tagMessage.setTagId(Long.valueOf(tagId));
                } else {
                    tagMessage.setSendAll(true);
                }
                wxMpService.getMassMessageService().massGroupMessageSend(tagMessage);
            }
        } catch (WxErrorException e) {
            logger.error("upload media file failed or send image message failed：", e);
            responseData.setMessage("upload media file failed or send image message failed：" + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    /**
     * 发送图片消息
     */
    @Override
    public ResponseData sendImage(IRequest requestContext, WxMessageBean messageBean) {
        int type = judgeType(messageBean.getAccountId(), messageBean.getAppId(), messageBean.getWxCode());
        if (type == INVALID_DATA) {
            ResponseData responseData = new ResponseData(false);
            responseData.setMessage("invalid param data value.");
            return responseData;
        } else {
            /*如果是企业号则调用企业号api*/
            if (type == CORP)
                return sendCpImage(messageBean);
                /*否则调用其他公众号接口*/
            else if (type == SERV || type == SUBS)
                return sendMpImage(messageBean);
            else {
                ResponseData responseData = new ResponseData(false);
                responseData.setMessage("invalid account type");
                return responseData;
            }
        }
    }

    /**
     * 上传图片并压缩
     */
    @Override
    public String uploadPicture(IRequest requestCtx, MultipartFile image) {
        String originalFileName = image.getOriginalFilename();
        String fileName = System.currentTimeMillis() + originalFileName;
        File dest = new File(uploadDir + fileName);
        if (WeChatUtils.compressImage(image, dest)) {
            return fileName;
        }
        return null;
    }

    /**
     * 非企业号上传图文消息缩略图并压缩
     */
    @Override
    public JSONObject uploadThumbPicture(IRequest requestCtx, MultipartFile image, Long accountId) {
        if (accountId == null) {
            logger.error("accountId can not be null");
            return null;
        }
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(accountId);
        if (account == null) {
            logger.error("target account can not be found");
            return null;
        }

        String originalFileName = image.getOriginalFilename();
        String fileName = System.currentTimeMillis() + originalFileName;
        File dest = new File(uploadDir + fileName);
        if (WeChatUtils.compressImage(image, dest)) {
            WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
            WxMediaUploadResult uploadMediaRes;
            try {
                uploadMediaRes = wxMpService.getMaterialService()
                        .mediaUpload(WxConsts.MassMsgType.IMAGE, FilenameUtils.getExtension(fileName), new FileInputStream(dest));
            } catch (WxErrorException | FileNotFoundException e) {
                logger.error("upload media file failed or send image message failed：", e);
                return null;
            }
            String mediaId = uploadMediaRes.getMediaId();
            String url = uploadDir + fileName;
            MPNewsThumbPicture mpNewsThumbPicture = new MPNewsThumbPicture();
            mpNewsThumbPicture.setOriginalName(originalFileName);
            mpNewsThumbPicture.setFileName(fileName);
            mpNewsThumbPicture.setUrl(url);
            mpNewsThumbPicture.setMediaId(mediaId);
            mpNewsThumbPicture.setAccountId(accountId);
            mpNewsThumbPictureMapper.insertSelective(mpNewsThumbPicture);
            JSONObject data = new JSONObject();
            data.put("mediaId", mediaId);
            data.put("fileName", fileName);
            return data;
        }
        return null;
    }

    /**
     * 企业号发送文本消息
     *
     * @param messageBean 消息bean
     * @return bool
     */
    private ResponseData sendCpText(WxMessageBean messageBean) {
        ResponseData responseData = new ResponseData(false);
        WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
        Integer agentId = Integer.valueOf(app.getAgentId());
        WxCpService wxCpService = WeChatUtils.initWxCpService(app.getCorpId(), app.getSecret(), Integer.valueOf(app.getAgentId()), proxyHost, proxyPort);
        String partyIds = formatIds(messageBean.getPartyIds(), true);
        String tagIds = formatIds(messageBean.getTagIds(), true);
        String userIds = formatIds(messageBean.getUserIds(), true);
        WxCpMessage wxCpMessage;
        if (StringUtils.isEmpty(partyIds) && StringUtils.isEmpty(tagIds) && StringUtils.isEmpty(userIds)) {
            wxCpMessage = WxCpMessage.TEXT()
                    .agentId(agentId)
                    .content(messageBean.getText())
                    .toUser("@all")
                    .build();
        } else {
            wxCpMessage = WxCpMessage.TEXT()
                    .agentId(agentId)
                    .content(messageBean.getText())
                    .toParty(partyIds)
                    .toTag(tagIds)
                    .toUser(userIds)
                    .build();
        }
        try {
            wxCpService.messageSend(wxCpMessage);
        } catch (WxErrorException e) {
            logger.error("send text message failed: ", e);
            responseData.setMessage("send text message failed: " + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    private WechatEnterpriseApp getWechatEnterpriseApp(WxMessageBean messageBean) {
        WechatEnterpriseApp app;
        if (StringUtils.isNotEmpty(messageBean.getWxCode())) {
            WechatEnterpriseApp dto = new WechatEnterpriseApp();
            dto.setWxCode(messageBean.getWxCode());
            app = wechatEnterpriseAppMapper.select(dto).get(0);
        } else {
            app = wechatEnterpriseAppMapper.selectByPrimaryKey(messageBean.getAppId());
        }
        return app;
    }

    /**
     * 非企业号发送文本消息
     *
     * @param messageBean 消息bean
     * @return bool
     */
    private ResponseData sendMpText(WxMessageBean messageBean) {
        ResponseData responseData = new ResponseData(false);
        WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        String[] openIds = formatIds(messageBean.getOpenIds());
        String tagId = messageBean.getTagId();
        try {
            if (ArrayUtils.isNotEmpty(openIds)) {
                WxMpMassOpenIdsMessage massMessage = new WxMpMassOpenIdsMessage();
                massMessage.setMsgType(WxConsts.MassMsgType.TEXT);
                massMessage.setContent(messageBean.getText());
                for (String openId : openIds) {
                    if (StringUtils.isNotEmpty(openId))
                        massMessage.getToUsers().add(openId);
                }
                wxMpService.getMassMessageService().massOpenIdsMessageSend(massMessage);
            } else {
                WxMpMassTagMessage tagMessage = new WxMpMassTagMessage();
                tagMessage.setMsgType(WxConsts.MassMsgType.TEXT);
                tagMessage.setContent(messageBean.getText());
                if (StringUtils.isNotEmpty(tagId)) {
                    tagMessage.setTagId(Long.valueOf(tagId));
                } else {
                    tagMessage.setSendAll(true);
                }
                wxMpService.getMassMessageService().massGroupMessageSend(tagMessage);
            }
        } catch (WxErrorException e) {
            logger.error("send text message failed: ", e);
            responseData.setMessage("send text message failed: " + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    /**
     * 发送文本消息
     */
    @Override
    public ResponseData sendText(IRequest requestContext, WxMessageBean messageBean) {
        int type = judgeType(messageBean.getAccountId(), messageBean.getAppId(), messageBean.getWxCode());
        if (type == INVALID_DATA) {
            ResponseData responseData = new ResponseData(false);
            responseData.setMessage("invalid param data value.");
            return responseData;
        } else if (type == CORP) {
            return sendCpText(messageBean);
        } else if (type == SERV || type == SUBS)
            return sendMpText(messageBean);
        else {
            ResponseData responseData = new ResponseData(false);
            responseData.setMessage("invalid account type");
            return responseData;
        }
    }

    /**
     * 企业号发送图文消息
     *
     * @param messageBean 消息bean
     * @return Boolean
     */
    private ResponseData sendCpMPNews(WxMessageBean messageBean) {
        ResponseData responseData = new ResponseData(false);
        WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
        Integer agentId = Integer.valueOf(app.getAgentId());
        WxCpService wxCpService = WeChatUtils.initWxCpService(app.getCorpId(), app.getSecret(), Integer.valueOf(app.getAgentId()), proxyHost, proxyPort);
        List<Article> articles = messageBean.getArticles();
        List<NewArticle> newArticles = new ArrayList<>();
        for (Article article : articles) {
            NewArticle newArticle = new NewArticle();
            newArticle.setDescription(article.getDescription());
            newArticle.setPicUrl(article.getPicUrl());
            newArticle.setTitle(article.getTitle());
            newArticle.setUrl(article.getUrl());
            newArticles.add(newArticle);
        }
        String partyIds = formatIds(messageBean.getPartyIds(), true);
        String tagIds = formatIds(messageBean.getTagIds(), true);
        String userIds = formatIds(messageBean.getUserIds(), true);
        WxCpMessage message;
        if (StringUtils.isEmpty(partyIds) && StringUtils.isEmpty(tagIds) && StringUtils.isEmpty(userIds)) {
            message = WxCpMessage.NEWS()
                    .toUser("@all")
                    .agentId(agentId)
                    .articles(newArticles)
                    .build();
        } else {
            message = WxCpMessage.NEWS()
                    .agentId(agentId)
                    .articles(newArticles)
                    .toParty(partyIds)
                    .toTag(tagIds)
                    .toUser(userIds)
                    .build();
        }
        try {
            wxCpService.messageSend(message);
        } catch (Exception e) {
            logger.error("send MPNews message failed: ", e);
            responseData.setMessage("send MPNews message failed: " + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    /**
     * 非企业号发送图文消息
     *
     * @param messageBean 消息bean
     * @return Boolean
     */
    private ResponseData sendMpNews(WxMessageBean messageBean) {
        ResponseData responseData = new ResponseData(false);
        WechatEnterpriseApp app = getWechatEnterpriseApp(messageBean);
        WechatAccount account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
        WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
        WxMpMassNews news = new WxMpMassNews();
        for (Article article : messageBean.getArticles()) {
            WxMpMassNews.WxMpMassNewsArticle newsArticle = new WxMpMassNews.WxMpMassNewsArticle();
            newsArticle.setAuthor(article.getAuthor());
            newsArticle.setContent(article.getContent());
            newsArticle.setContentSourceUrl(article.getContentSourceUrl());
            newsArticle.setDigest(article.getDescription());
            newsArticle.setThumbMediaId(article.getThumbMediaId());
            newsArticle.setShowCoverPic(article.getShowCoverPic());
            newsArticle.setTitle(article.getTitle());
            news.addArticle(newsArticle);
        }
        String[] openIds = formatIds(messageBean.getOpenIds());
        String tagId = messageBean.getTagId();
        try {
            WxMpMassUploadResult massUploadResult = wxMpService.getMassMessageService().massNewsUpload(news);
            if (ArrayUtils.isNotEmpty(openIds)) {
                WxMpMassOpenIdsMessage massMessage = new WxMpMassOpenIdsMessage();
                massMessage.setMsgType(WxConsts.MassMsgType.MPNEWS);
                massMessage.setMediaId(massUploadResult.getMediaId());
                for (String openId : openIds) {
                    massMessage.getToUsers().add(openId);
                }
                wxMpService.getMassMessageService().massOpenIdsMessageSend(massMessage);
            } else {
                WxMpMassTagMessage tagMessage = new WxMpMassTagMessage();
                tagMessage.setMsgType(WxConsts.MassMsgType.MPNEWS);
                tagMessage.setMediaId(massUploadResult.getMediaId());
                if (StringUtils.isNotEmpty(tagId)) {
                    tagMessage.setTagId(Long.valueOf(tagId));
                } else {
                    tagMessage.setSendAll(true);
                }
                wxMpService.getMassMessageService().massGroupMessageSend(tagMessage);
            }

        } catch (WxErrorException e) {
            logger.error("send MPNews message failed: ", e);
            responseData.setMessage("send MPNews message failed: " + e.getMessage());
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    /**
     * 发送图文消息
     */
    @Override
    public ResponseData sendMPNews(IRequest iRequest, WxMessageBean messageBean) {
        int type = judgeType(messageBean.getAccountId(), messageBean.getAppId(), messageBean.getWxCode());
        if (type == INVALID_DATA) {
            ResponseData responseData = new ResponseData(false);
            responseData.setMessage("invalid param data value.");
            return responseData;
        } else if (type == CORP) {
            return sendCpMPNews(messageBean);
        } else if (type == SERV || type == SUBS)
            return sendMpNews(messageBean);
        else {
            ResponseData responseData = new ResponseData(false);
            responseData.setMessage("invalid account type");
            return responseData;
        }
    }

    /**
     * 根据accountId和appId判断信息可用性和公众号类型
     *
     * @param accountId 公众号id
     * @param appId     企业号应用id
     * @return -1表示信息不可用，1表示企业号并有对应的应用，2表示订阅号，3表示服务号
     */
    @Override
    public Integer judgeType(Long accountId, Long appId, String wxCode) {
        WechatAccount account;
        WechatEnterpriseApp app;
        Integer accountType;
        if (StringUtils.isNotEmpty(wxCode)) {
            WechatEnterpriseApp wea = new WechatEnterpriseApp();
            wea.setWxCode(wxCode);
            List<WechatEnterpriseApp> apps = wechatEnterpriseAppMapper.select(wea);
            if (CollectionUtils.isEmpty(apps) || apps.size() > 1) {
                logger.error("one wxCode found duplicate app.");
                return INVALID_DATA;
            }
            app = apps.get(0);
            account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
            if (account == null) {
                logger.error("app has no account.");
                return INVALID_DATA;
            }
            return account.getAccountType();
        } else if (accountId != null) {
            account = wechatAccountMapper.selectByPrimaryKey(accountId);
            if (account == null) {
                if (appId != null) {
                    app = wechatEnterpriseAppMapper.selectByPrimaryKey(appId);
                    if (app == null) {
                        logger.error("target wechat enterprise app can not be found!");
                        return INVALID_DATA;
                    }
                    return CORP;
                }
                logger.error("target wechat account can not be found!");
                return INVALID_DATA;
            } else {
                accountType = account.getAccountType();
                if (!accountType.equals(CORP)
                        && !accountType.equals(SERV)
                        && !accountType.equals(SUBS))
                    return INVALID_DATA;
                else if (accountType.equals(CORP)) {
                    if (appId == null) {
                        logger.error("appId can not be null when your account type is qiye");
                        return INVALID_DATA;
                    } else {
                        app = wechatEnterpriseAppMapper.selectByPrimaryKey(appId);
                        if (app == null) {
                            logger.error("target wechat enterprise app can not be found!");
                            return INVALID_DATA;
                        }
                        return CORP;
                    }
                } else
                    return accountType;
            }
        } else {
            if (appId == null) {
                logger.error("appId & accountId can not both be null");
                return INVALID_DATA;
            } else {
                app = wechatEnterpriseAppMapper.selectByPrimaryKey(appId);
                if (app == null) {
                    logger.error("target wechat app can not be found!");
                    return INVALID_DATA;
                }
                account = wechatAccountMapper.selectByPrimaryKey(app.getAccountId());
                if (account == null) {
                    logger.error("target wechat account can not be found!");
                    return INVALID_DATA;
                }
                if (!account.getAccountType().equals(CORP)
                        && !account.getAccountType().equals(SERV)
                        && !account.getAccountType().equals(SUBS))
                    return INVALID_DATA;
                else
                    return account.getAccountType();
            }
        }
    }
}
