package hls.support.core.wechat.service.impl;

import com.hand.hap.core.IRequest;
import com.hand.hap.system.dto.ResponseData;
import com.hand.hap.system.service.impl.BaseServiceImpl;
import hls.support.core.wechat.dto.WechatAccount;
import hls.support.core.wechat.dto.WechatEnterpriseApp;
import hls.support.core.wechat.dto.WechatMenu;
import hls.support.core.wechat.mapper.WechatAccountMapper;
import hls.support.core.wechat.mapper.WechatEnterpriseAppMapper;
import hls.support.core.wechat.mapper.WechatMenuMapper;
import hls.support.core.wechat.service.IWechatMenuService;
import hls.support.core.wechat.utils.WeChatUtils;
import me.chanjar.weixin.common.bean.menu.WxMenu;
import me.chanjar.weixin.common.bean.menu.WxMenuButton;
import me.chanjar.weixin.common.exception.WxErrorException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.mp.api.WxMpService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.cxf.common.util.UrlUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.*;

@Service
@Transactional
public class WechatMenuServiceImpl extends BaseServiceImpl<WechatMenu> implements IWechatMenuService {
    private String qyMenuApi = "https://qyapi.weixin.qq.com/cgi-bin/menu/create?access_token={ACCESS_TOKEN}&agentid={AGENTID}";

    private static Logger logger = LoggerFactory.getLogger(WechatMenuServiceImpl.class);
    @Autowired
    private WechatEnterpriseAppMapper appMapper;
    @Autowired
    private WechatAccountMapper accountMapper;
    @Autowired
    private WechatMenuMapper menuMapper;
    @Value("#{configProperties['http.proxy.port']}")
    private Integer proxyPort;

    @Value("#{configProperties['http.proxy.host']}")
    private String proxyHost;

    private static final String urlPrefix = "https://open.weixin.qq.com/connect/oauth2/authorize?";
    private static final String responseType = "code";
    private static final String urlSuffix = "#wechat_redirect";
    private static final String encodeTypeUtf8 = "utf-8";
    private static final String buttonTypeView = "view";

    private boolean validateData(WechatMenu dto) {
        Long wechatAccountId = dto.getWechatAccountId();
        if (wechatAccountId == null) {
            logger.error("app id is null");
            return false;
        }
        WechatEnterpriseApp app = appMapper.selectByPrimaryKey(wechatAccountId);
        if (app == null) {
            logger.error("target app not found.");
            return false;
        }
        WechatAccount account = accountMapper.selectByPrimaryKey(app.getAccountId());
        if (account == null) {
            logger.error("target account not found.");
            return false;
        }
        return true;
    }

    private Map<String, Object> getCorpMenus(Long accountId) {
        Map<String, Object> map = new HashMap<>(1);
        List<Map<String, Object>> buttons = new LinkedList<>();
        map.put("buttons", buttons);
        List<WechatMenu> menus = menuMapper.selectMenusByAccountId(accountId);
        if (menus == null || menus.size() < 1) {
            return map;
        }
        List<WechatMenu> mains = new LinkedList<>();
        HashMap<Long, List<Map<String, Object>>> subs = new HashMap<>(menus.size());

        for (WechatMenu menu : menus) {
            Long parentMenuId = menu.getParentMenuId();
            if (parentMenuId != null) {
                // 子菜单
                List<Map<String, Object>> list = subs.containsKey(parentMenuId) ? subs.get(parentMenuId) : new LinkedList<>();
                list.add(parse2Map(menu, false));
                subs.put(parentMenuId, list);
            } else {
                // 主菜单
                mains.add(menu);
            }
        }

        if (mains.size() < 1) {
            return map;
        }
        // 准备格式化
        for (WechatMenu main : mains) {
            if (subs.containsKey(main.getId())) {
                // 有子菜单
                Map<String, Object> menu = parse2Map(main, true);
                menu.put("sub_button", subs.get(main.getId()));
                buttons.add(menu);
            } else {
                // 无子菜单
                Map<String, Object> menu = parse2Map(main, false);
                buttons.add(menu);
            }
        }

        return map;
    }

    private Map<String, Object> parse2Map(WechatMenu menu, boolean hasChild) {
        if (menu == null) {
            return new HashMap<>(0);
        }
        Map<String, Object> map = new HashMap<>(5);
        map.put("name", menu.getName());
        if (!hasChild) {
            map.put("type", menu.getType());
            if ("view".equalsIgnoreCase(menu.getType())) {
                map.put("url", menu.getUrl());
            } else {
                map.put("key", menu.getMenuKey());
            }
        }
        return map;
    }

    @Override
    public ResponseData addMenu(IRequest requestCtx, WechatMenu dto) {
        ResponseData responseData = new ResponseData(false);
        if (validateData(dto)) {
            dto.setIsPublished("N");
            try {
                menuMapper.insertSelective(dto);
                Long parentMenuId = dto.getId();
                if (parentMenuId != null) {
                    List<WechatMenu> childMenus = dto.getChildMenus();
                    if (childMenus != null) {
                        for (WechatMenu child : childMenus) {
                            child.setParentMenuId(parentMenuId);
                            child.setWechatAccountId(dto.getWechatAccountId());
                            child.setIsPublished("N");
                            menuMapper.insertSelective(child);
                        }
                    }
                }
            } catch (Exception e) {
                logger.error("insert menu data failed,", e);
                responseData.setMessage("insert menu data failed.");
                return new ResponseData(false);
            }
        } else {
            responseData.setMessage("request data validate error.");
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    @Override
    public ResponseData updateMenu(IRequest requestCtx, List<WechatMenu> wechatMenus) {
        ResponseData responseData = new ResponseData(false);
        if (CollectionUtils.isNotEmpty(wechatMenus)) {
            for (WechatMenu dto : wechatMenus) {
                if (validateData(dto)) {
                    dto = setUrl(dto);
                    if (dto == null) {
                        responseData.setMessage("set url for menu error");
                        return responseData;
                    }
                    dto.setIsPublished("N");
                    List<WechatMenu> childMenus = dto.getChildMenus();
                    if (dto.getId() != null) {
                        menuMapper.updateByPrimaryKeySelective(dto);
                    } else {
                        menuMapper.insertSelective(dto);
                    }
                    WechatMenu menu = new WechatMenu();
                    menu.setParentMenuId(dto.getId());
                    menuMapper.delete(menu);
                    if (childMenus != null) {
                        for (WechatMenu child : childMenus) {
                            child.setWechatAccountId(dto.getWechatAccountId());
                            child = setUrl(child);
                            if (child == null) {
                                responseData.setMessage("set url for menu error.");
                                return responseData;
                            }
                            child.setParentMenuId(dto.getId());
                            child.setWechatAccountId(dto.getWechatAccountId());
                            child.setIsPublished("N");
                            menuMapper.insertSelective(child);
                        }
                    }
                } else {
                    responseData.setMessage("request data validate error.");
                    return responseData;
                }
            }
        }
        responseData.setSuccess(true);
        return responseData;
    }

    @Override
    public ResponseData deleteMenu(IRequest requestCtx, WechatMenu dto) {
        ResponseData responseData = new ResponseData(false);
        WechatMenu parent = menuMapper.selectByPrimaryKey(dto);
        if (parent == null) {
            responseData.setMessage("target menu not found.");
            return responseData;
        }
        WechatMenu menu = new WechatMenu();
        menu.setParentMenuId(parent.getId());
        try {
            menuMapper.delete(menu);
            menuMapper.deleteByPrimaryKey(parent);
        } catch (Exception e) {
            logger.error("delete menus failed,", e);
            responseData.setMessage("delete menus failed," + e);
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    @Override
    public ResponseData queryMenu(IRequest requestCtx, WechatMenu dto) {
        ResponseData responseData = new ResponseData(false);
        if (validateData(dto)) {
            List<WechatMenu> menus = menuMapper.select(dto);
            List<WechatMenu> parents = new ArrayList<>();
            List<WechatMenu> children = new ArrayList<>();
            for (WechatMenu menu : menus) {
                if (menu.getParentMenuId() == null) {
                    parents.add(menu);
                } else {
                    children.add(menu);
                }
            }
            for (WechatMenu parent : parents) {
                for (Iterator iterator = children.iterator(); iterator.hasNext(); ) {
                    WechatMenu child = (WechatMenu) iterator.next();
                    if (child.getParentMenuId().equals(parent.getId())) {
                        List<WechatMenu> childMenus = parent.getChildMenus();
                        if (childMenus == null) {
                            childMenus = new ArrayList<>();
                        }
                        childMenus.add(child);
                        parent.setChildMenus(childMenus);
                        iterator.remove();
                    }
                }
            }
            responseData.setSuccess(true);
            responseData.setRows(parents);
            return responseData;
        } else {
            responseData.setMessage("request data validate error.");
            return responseData;
        }
    }

    @Override
    public ResponseData synchronizeMenu(IRequest requestCtx, WechatMenu dto) {
        ResponseData responseData = new ResponseData(false);
        Long wechatAccountId = dto.getWechatAccountId();
        if (wechatAccountId == null) {
            responseData.setMessage("app id is null");
            return responseData;
        }
        WechatEnterpriseApp app = appMapper.selectByPrimaryKey(wechatAccountId);
        if (app == null) {
            responseData.setMessage("target app not found.");
            return responseData;
        }
        WechatAccount account = accountMapper.selectByPrimaryKey(app.getAccountId());
        if (account == null) {
            responseData.setMessage("target account not found.");
            return responseData;
        }
        Integer accountType = account.getAccountType();
        List<WxMenuButton> buttons = null;
        if (accountType.equals(WechatAccount.CORP)) {
            WxCpService wxCpService = WeChatUtils.initWxCpService(account.getAppId(), app.getSecret(), Integer.valueOf(app.getAgentId()), proxyHost, proxyPort);
            try {
                WxMenu wxMenu = wxCpService.getMenuService().get();
                buttons = wxMenu.getButtons();
            } catch (WxErrorException e) {
                logger.error("get menus data from wechat server failed.", e);
                responseData.setMessage("get menus data from wechat server failed." + e.getMessage());
                return responseData;
            }
        } else if (accountType.equals(WechatAccount.SERV) || accountType.equals(WechatAccount.SUBS)) {
            WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), proxyHost, proxyPort);
            try {
                buttons = wxMpService.getMenuService().menuGet().getMenu().getButtons();
            } catch (WxErrorException e) {
                logger.error("get menus data from wechat server failed.", e);
                responseData.setMessage("get menus data from wechat server failed." + e.getMessage());
                return responseData;
            }
        }
        WechatMenu menu = new WechatMenu();
        menu.setWechatAccountId(app.getId());
        try {
            menuMapper.delete(menu);
            if (CollectionUtils.isNotEmpty(buttons)) {
                for (WxMenuButton button : buttons) {
                    Map<String, String> map = resolveUrl(button.getUrl());
                    WechatMenu parent = new WechatMenu();
                    BeanUtils.copyProperties(button, parent);
                    parent.setWechatAccountId(app.getId());
                    parent.setMenuKey(button.getKey());
                    if (map != null) {
                        parent.setIsPublished("Y");
                        parent.setRedirectUri(map.get("redirect_uri"));
                        parent.setScope(map.get("scope"));
                        parent.setState(Integer.valueOf(StringUtils.split(map.get("state"), "#")[0]));
                    }
                    menuMapper.insertSelective(parent);
                    List<WxMenuButton> subButtons = button.getSubButtons();
                    if (CollectionUtils.isNotEmpty(subButtons)) {
                        for (WxMenuButton subButton : subButtons) {
                            Map<String, String> childMap = resolveUrl(subButton.getUrl());
                            if (childMap != null) {
                                WechatMenu child = new WechatMenu();
                                BeanUtils.copyProperties(subButton, child);
                                child.setWechatAccountId(app.getId());
                                child.setMenuKey(subButton.getKey());
                                child.setParentMenuId(parent.getId());
                                child.setIsPublished("Y");
                                child.setState(Integer.valueOf(StringUtils.split(childMap.get("state"), "#")[0]));
                                child.setScope(childMap.get("scope"));
                                child.setRedirectUri(childMap.get("redirect_uri"));
                                menuMapper.insertSelective(child);
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            logger.error("refresh menu data in database failed,", e);
            responseData.setMessage("refresh menu data in database failed.");
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    @Override
    public ResponseData publishMenu(IRequest requestCtx, WechatMenu dto) {
        ResponseData responseData = new ResponseData(false);
        Long wechatAccountId = dto.getWechatAccountId();
        if (wechatAccountId == null) {
            responseData.setMessage("app id is null");
            return responseData;
        }
        WechatEnterpriseApp app = appMapper.selectByPrimaryKey(wechatAccountId);
        if (app == null) {
            responseData.setMessage("target app not found.");
            return responseData;
        }
        WechatAccount account = accountMapper.selectByPrimaryKey(app.getAccountId());
        if (account == null) {
            responseData.setMessage("target account not found.");
            return responseData;
        }
        Integer accountType = account.getAccountType();
        WechatMenu menu = new WechatMenu();
        menu.setWechatAccountId(app.getId());
        List<WechatMenu> menus = menuMapper.select(menu);
        List<WechatMenu> parents = new ArrayList<>();
        List<WechatMenu> children = new ArrayList<>();
        for (WechatMenu m : menus) {
            m.setIsPublished("Y");
            menuMapper.updateByPrimaryKeySelective(m);
            if (m.getParentMenuId() == null) {
                parents.add(m);
            } else {
                children.add(m);
            }
        }
        WxMenu wxMenu = new WxMenu();
        List<WxMenuButton> buttons = new ArrayList<>();
        for (WechatMenu parent : parents) {
            WxMenuButton button = new WxMenuButton();
            BeanUtils.copyProperties(parent, button);
            button.setType(buttonTypeView);
            button.setKey(parent.getMenuKey());
            if (accountType.equals(WechatAccount.SERV) || accountType.equals(WechatAccount.SUBS)) {
                button.setAppId(account.getAppId());
            }
            List<WxMenuButton> subButtons = new ArrayList<>();
            for (WechatMenu child : children) {
                if (child.getParentMenuId().equals(parent.getId())) {
                    WxMenuButton subButton = new WxMenuButton();
                    BeanUtils.copyProperties(child, subButton);
                    subButton.setKey(child.getMenuKey());
                    subButton.setType(buttonTypeView);
                    if (accountType.equals(WechatAccount.SERV) || accountType.equals(WechatAccount.SUBS)) {
                        subButton.setAppId(account.getAppId());
                    }
                    subButtons.add(subButton);
                }
            }
            button.setSubButtons(subButtons);
            buttons.add(button);
        }
        wxMenu.setButtons(buttons);
        if (accountType.equals(WechatAccount.CORP)) {
            WxCpService wxCpService = WeChatUtils.initWxCpService(account.getAppId(), app.getSecret(), Integer.valueOf(app.getAgentId()), proxyHost, proxyPort);
            try {
                wxCpService.getMenuService().create(wxMenu);
            } catch (WxErrorException e) {
                logger.error("create menu failed,", e);
                responseData.setMessage("create menu failed.");
                return responseData;
            }
        } else if (accountType.equals(WechatAccount.SERV) || accountType.equals(WechatAccount.SUBS)) {
            WxMpService wxMpService = WeChatUtils.initWxMpService(account.getAppId(), account.getSecret(), null, null);
            try {
                wxMpService.getMenuService().menuCreate(wxMenu);
            } catch (WxErrorException e) {
                logger.error("create menu failed,", e);
                responseData.setMessage("create menu failed.");
                return responseData;
            }
        } else {
            responseData.setMessage("invalid account type.");
            return responseData;
        }
        responseData.setSuccess(true);
        return responseData;
    }

    @Override
    public ResponseData pushMenus(Long accountId) {
        return null;
    }

    private WechatMenu setUrl(WechatMenu dto) {
        Long wechatAccountId = dto.getWechatAccountId();
        if (wechatAccountId == null) {
            logger.error("wechat menu's wechatAccountId can not be null.");
            return null;
        }
        String url = generateUrl(wechatAccountId, dto.getRedirectUri(), dto.getScope(), dto.getState());
        if (StringUtils.isNotEmpty(url)) {
            dto.setUrl(url);
        }
        return dto;
    }

    /**
     * 根据参数生成菜单url
     *
     * @param wecchatAccountId 微信应用id
     * @param redirectUri      重定向url
     * @param scope            权限域
     * @param state            state
     * @return 生成的url string
     */
    private String generateUrl(Long wecchatAccountId, String redirectUri, String scope, Integer state) {
        if (StringUtils.isEmpty(redirectUri) || StringUtils.isEmpty(scope) || state == null) {
            return null;
        }
        WechatEnterpriseApp app = appMapper.selectByPrimaryKey(wecchatAccountId);
        if (app == null) {
            logger.error("app not found.");
            return null;
        }
        WechatAccount account = accountMapper.selectByPrimaryKey(app.getAccountId());
        if (account == null) {
            logger.error("account not found.");
            return null;
        }
        String encodeUri;
        try {
            encodeUri = URLEncoder.encode(redirectUri, encodeTypeUtf8);
        } catch (UnsupportedEncodingException e) {
            logger.error("encode redirectUri failed,", e);
            return null;
        }
        return urlPrefix + "appid=" + account.getAppId() +
                "&redirect_uri=" + encodeUri +
                "&response_type=" + responseType +
                "&scope=" + scope +
                "&state=" + state +
                urlSuffix;
    }

    /**
     * 解析同步到的url数据
     *
     * @param url 微信服务器button的url
     * @return map
     */
    private Map<String, String> resolveUrl(String url) {
        if (StringUtils.isEmpty(url)) {
            return null;
        }
        try {
            url = URLDecoder.decode(url, encodeTypeUtf8);
        } catch (UnsupportedEncodingException e) {
            logger.error("url decode error", e);
        }
        return UrlUtils.parseQueryString(url);
    }

}
