package com.hand.app.esign.utils;

import com.alibaba.fastjson.JSON;
import com.timevale.esign.sdk.tech.bean.*;
import com.timevale.esign.sdk.tech.bean.result.*;
import com.timevale.esign.sdk.tech.bean.seal.OrganizeTemplateType;
import com.timevale.esign.sdk.tech.bean.seal.PersonTemplateType;
import com.timevale.esign.sdk.tech.bean.seal.SealColor;
import com.timevale.esign.sdk.tech.impl.constants.LegalAreaType;
import com.timevale.esign.sdk.tech.impl.constants.LicenseQueryType;
import com.timevale.esign.sdk.tech.impl.constants.SignType;
import com.timevale.esign.sdk.tech.service.*;
import com.timevale.esign.sdk.tech.service.factory.*;
import com.timevale.tech.sdk.bean.HttpConnectionConfig;
import com.timevale.tech.sdk.bean.ProjectConfig;
import com.timevale.tech.sdk.bean.SignatureConfig;
import com.timevale.tech.sdk.constants.AlgorithmType;
import com.timevale.tech.sdk.constants.HttpType;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created with IntelliJ IDEA.
 * User: yang
 * Date: 2018/1/18
 * Time: 11:13
 */
public class SignUtils {

    private static final int DEFAULT_RETRY_TIME = 5;
    public static final int DEFAULT_USER_SIGN_WIDTH = 100;
    public static final int DEFAULT_PLAT_SIGN_WIDTH = 159;
    public static final int POS_TYPE_POSTION = 0;
    public static final int POS_TYPE_KEYWORD = 1;
    public static final int CODE_SUCCESS = 0;
    public static final int CODE_ERROR_ACCOUNT_EXIST = 150016;
    public static final int DEFAULT_PLATFORM_SEAL_ID = 0;

    private static Logger logger = LoggerFactory.getLogger(SignUtils.class);

    /***
     * 项目初始化 使用到的接口：sdk.init(proCfg, httpConCfg, sCfg);
     * @param projectId
     * @param projectSecret
     * @param apiUrl
     */
    public static int initProject(String projectId, String projectSecret, String apiUrl) {

        ProjectConfig proCfg = initProjectConfig(projectId, projectSecret, apiUrl);
        HttpConnectionConfig httpConCfg = initHttpConnectionConfig();
        SignatureConfig sCfg = initSignatureConfig();

        logger.info("eSign init...");
        EsignsdkService sdk = EsignsdkServiceFactory.instance();
        Result result = sdk.init(proCfg, httpConCfg, sCfg);
        if (CODE_SUCCESS != result.getErrCode()) {
            logger.error("eSign init failed：errCode={},msg={}", result.getErrCode(), result.getMsg());
        } else {
            logger.info("eSign init success：errCode={},msg={}", result.getErrCode(), result.getMsg());
        }
        return result.getErrCode();
    }

    /***
     * 用户关键字定位签署的PosBean
     */
    public static PosBean getUserKeywordPosBean(String keyword) {
        return getKeywordPosBean(keyword, DEFAULT_USER_SIGN_WIDTH);
    }

    /***
     * 企业用户关键字定位签署的PosBean
     */
    public static PosBean getOrganizeKeywordPosBean(String keyword) {
        return getKeywordPosBean(keyword, DEFAULT_USER_SIGN_WIDTH);
    }

    /***
     * 平台关键字定位签署的PosBean
     */
    public static PosBean getPlatformKeywordPosBean(String keyword) {
        return getKeywordPosBean(keyword, DEFAULT_PLAT_SIGN_WIDTH);
    }

    /***
     * 关键字定位签署的PosBean
     */
    public static PosBean getKeywordPosBean(String keyword, int width) {
        PosBean posBean = new PosBean();
        posBean.setPosType(POS_TYPE_KEYWORD);
        posBean.setKey(keyword);
        posBean.setWidth(width);
        return posBean;
    }

    /***
     * 用户坐标定位签署的PosBean
     */
    public static PosBean getUserXYPosBean(String page, int x, int y) {
        return getXYPosBean(page, x, y, DEFAULT_USER_SIGN_WIDTH);
    }

    /***
     * 平台坐标定位签署的PosBean
     */
    public static PosBean getPlatXYPosBean(String page, int x, int y) {
        return getXYPosBean(page, x, y, DEFAULT_PLAT_SIGN_WIDTH);
    }

    /***
     * 坐标定位签署的PosBean
     */
    public static PosBean getXYPosBean(String page, int x, int y, int width) {
        PosBean posBean = new PosBean();
        posBean.setPosType(POS_TYPE_POSTION);
        posBean.setPosPage(page);
        posBean.setPosX(x);
        posBean.setPosY(y);
        posBean.setWidth(width);
        return posBean;
    }

    /***
     * 文件流签署的PDF文档信息
     */
    private static SignPDFStreamBean setSignPDFStreamBean(byte[] pdfData) {
        SignPDFStreamBean signPDFStreamBean = new SignPDFStreamBean();
        signPDFStreamBean.setStream(pdfData);
        return signPDFStreamBean;
    }

    /***
     * 文件路径签署的PDF文档信息
     */
    public static SignPDFFileBean setSignPDFFileBean(String srcPdfFile, String signedPdfFile, String signedPdfFileName) {
        SignPDFFileBean signPDFFileBean = new SignPDFFileBean();
        // 待签署文档本地路径
        signPDFFileBean.setSrcPdfFile(srcPdfFile);
        // 签署后文档本地路径
        signPDFFileBean.setDstPdfFile(signedPdfFile);
        // 文档名称，e签宝签署日志对应的文档名，若为空则取文档路径中的名称
        signPDFFileBean.setFileName(signedPdfFileName);
        // 文档编辑密码，当目标PDF设置权限密码保护时必填 */
        signPDFFileBean.setOwnerPassword(null);
        return signPDFFileBean;
    }

    /**
     * 用户摘要签署
     *
     * @param accountId 用户id
     * @param pdfData   文档内容
     * @param sealData  base64签章内容
     * @param page      签署页面
     * @param x         x坐标
     * @param y         y坐标
     * @return 签署结果
     */
    public static FileDigestSignResult customerDigestSign(byte[] pdfData, String accountId, String sealData,
                                                          String page, int x, int y) {
        // 设置文件流签署的PDF文档信息
        SignPDFStreamBean signPDFStreamBean = setSignPDFStreamBean(pdfData);
        // 设置坐标定位签署的PosBean，坐标定位方式支持单页签章、多页签章和骑缝章，但对关键字签章指定页码无效；
        PosBean posBean = getUserXYPosBean(page, x, y);
        SignType signType = SignType.Single;
        if (StringUtils.containsAny(page, '-', ',')) {
            signType = SignType.Multi;
        }
        return customerDigestSign(signPDFStreamBean, accountId, sealData, posBean, signType);
    }

    /**
     * 用户摘要签署
     *
     * @param accountId 用户id
     * @param pdfData   文档内容
     * @param sealData  base64签章内容
     * @return 签署结果
     */
    public static FileDigestSignResult customerDigestSign(byte[] pdfData, String accountId, String sealData,
                                                          PosBean posBean, SignType signType) {
        // 设置文件流签署的PDF文档信息
        SignPDFStreamBean signPDFStreamBean = setSignPDFStreamBean(pdfData);
        return customerDigestSign(signPDFStreamBean, accountId, sealData, posBean, signType);
    }

    /**
     * 用户摘要签署 文件
     *
     * @param accountId 用户id
     * @param srcPdfFile   待签署文档本地路径
     * @param signedPdfFile   签署后文档本地路径
     * @param signedPdfFileName   文档名称，e签宝签署日志对应的文档名，若为空则取文档路径中的名称
     * @param sealData  base64签章内容
     * @return 签署结果
     */
    public static FileDigestSignResult userOrganizeSignByFile(String srcPdfFile, String signedPdfFile, String signedPdfFileName, String accountId, String sealData,
                                                          PosBean posBean, SignType signType) {
        // 设置文件流签署的PDF文档信息
        SignPDFFileBean setSignPDFFileBean = setSignPDFFileBean(srcPdfFile,signedPdfFile,signedPdfFileName);

        return userOrganizeSignByStream(setSignPDFFileBean, accountId, sealData, posBean, signType);
    }

    /***
     * 平台下用户PDF摘要签署（文件）；盖章位置通过关键字定位； 使用到接口：UserSignServiceFactory.instance();
     * userSignService.localSignPDF(accountId,addSealResult.getSealData(),
     * signPDFStreamBean, posBean, SignType.Single);
     */
    public static FileDigestSignResult userOrganizeSignByStream(SignPDFFileBean signPDFFileBean, String accountId,
                                                                String sealData, PosBean posBean, SignType signType) {
        logger.info("user sign start.");
        if (logger.isDebugEnabled()) {
            logger.debug("accountId: {},posBean: {},sealData: {},signType: {}", accountId, posBean, sealData, signType);
        }
        if (anyNullOrBlank(signPDFFileBean, accountId, sealData, posBean, signType)) {
            throw new IllegalArgumentException("one or more args are null");
        }
        UserSignService userSignService = UserSignServiceFactory.instance();
        FileDigestSignResult result = userSignService.localSignPDF(accountId, sealData, signPDFFileBean, posBean, signType);
        if (logger.isDebugEnabled()) {
            logger.debug(JSON.toJSONString(result));
        }
        return result;
    }



        /**
         * 用户摘要签署
         *
         * @param accountId         用户id
         * @param signPDFStreamBean 文档内容
         * @param sealData          base64签章内容
         * @return 签署结果
         */
    public static FileDigestSignResult customerDigestSign(SignPDFStreamBean signPDFStreamBean, String accountId,
                                                          String sealData, PosBean posBean, SignType signType) {

        logger.info("customer sign start.");
        if (logger.isDebugEnabled()) {
            logger.debug("accountId: {},posBean: {},sealData: {},signType: {}", accountId, posBean, sealData, signType);
        }
        if (anyNullOrBlank(signPDFStreamBean, accountId, sealData, posBean, signType)) {
            throw new IllegalArgumentException("one or more args are null");
        }
        UserSignService userSignService = UserSignServiceFactory.instance();
        FileDigestSignResult result = userSignService.localSignPDF(accountId, sealData, signPDFStreamBean, posBean, signType);
        if (logger.isDebugEnabled()) {
            logger.debug(JSON.toJSONString(result));
        }
        return result;
    }


    /**
     * 使用默认签章进行平台签署
     */
    public static FileDigestSignResult platformSignByStreamm(byte[] pdfData, String page, int x, int y) {
        // 设置文件流签署的PDF文档信息
        SignPDFStreamBean signPDFStreamBean = setSignPDFStreamBean(pdfData);
        PosBean posBean = getPlatXYPosBean(page, x, y);
        SignType signType = SignType.Single;
        if (StringUtils.containsAny(page, '-', ',')) {
            signType = SignType.Multi;
        }
        // 设置签署印章，www.tsign.cn官网设置的默认签名sealId = 0
        int sealId = 0;
        logger.info("platform self sign use default seal");
        if (logger.isDebugEnabled()) {
            logger.debug("page:{},x:{},y:{}", page, x, y);
        }

        logger.info("----platform self sign by digest start...");
        SelfSignService selfSignService = SelfSignServiceFactory.instance();
        FileDigestSignResult result = selfSignService.localSignPdf(signPDFStreamBean, posBean, sealId, signType);
        if (CODE_SUCCESS != result.getErrCode()) {
            logger.info("platform self sign by digest failed，errCode={},msg={}", result.getErrCode(), result.getMsg());
        } else {
            logger.info("platform self sign by digest success, SignServiceId ={} ", result.getSignServiceId());
        }
        return result;

    }

    /**
     * 使用默认签章进行平台签署
     */
    public static FileDigestSignResult platformSignByStreamm(byte[] pdfData, PosBean posBean, SignType signType) {
        // 设置文件流签署的PDF文档信息
        SignPDFStreamBean signPDFStreamBean = setSignPDFStreamBean(pdfData);
        return platformSignByStreamm(signPDFStreamBean, posBean, signType);

    }

    /**
     * 使用默认签章进行平台签署
     */
    public static FileDigestSignResult platformSignByStreamm(SignPDFStreamBean pdfData, PosBean posBean, SignType signType) {

        logger.info("----platform self sign by digest start...");
        SelfSignService selfSignService = SelfSignServiceFactory.instance();
        FileDigestSignResult result = selfSignService.localSignPdf(pdfData, posBean, DEFAULT_PLATFORM_SEAL_ID, signType);
        if (CODE_SUCCESS != result.getErrCode()) {
            logger.info("platform self sign by digest failed，errCode={},msg={}", result.getErrCode(), result.getMsg());
        } else {
            logger.info("platform self sign by digest success, SignServiceId ={} ", result.getSignServiceId());
        }
        return result;

    }

    /**
     * 使用指定签章进行平台签署
     */
    public static FileDigestSignResult platformSignByStreamm(SignPDFStreamBean pdfData, PosBean posBean, SignType signType, int sealId) {

        logger.info("----platform self sign by digest start...");
        SelfSignService selfSignService = SelfSignServiceFactory.instance();
        FileDigestSignResult result = selfSignService.localSignPdf(pdfData, posBean, sealId, signType);
        if (CODE_SUCCESS != result.getErrCode()) {
            logger.info("platform self sign by digest failed，errCode={},msg={}", result.getErrCode(), result.getMsg());
        } else {
            logger.info("platform self sign by digest success, SignServiceId ={} ", result.getSignServiceId());
        }
        return result;

    }

    public static GetAccountProfileResult getExistUserAccount(String idNo) {
        return getExistUserAccount(idNo, LicenseQueryType.MAINLAND);
    }

    public static GetAccountProfileResult getExistUserAccount(String idNo, LicenseQueryType area) {
        AccountService accountService = AccountServiceFactory.instance();
        return accountService.getAccountInfoByIdNo(idNo, area);
    }

    /***
     * 创建个人账户 使用到接口：accountService.addAccount(organizeBean);
     * @param name 姓名
     * @param idNo 身份证号
     * @param area 个人归属地：0-大陆，1-香港，2-澳门，3-台湾，4-外籍，默认0
     * @param email 邮箱，可空
     * @param mobile 手机号，可空
     * @param organ 所属公司,可空
     * @param title 职位,可空
     * @param address 常用地址,可空
     *
     * @return 用户的accountId
     */
    public static AddAccountResult addPersonAccount(String name, String idNo, int area, String email, String mobile, String organ, String title, String address) {
        PersonBean personBean = new PersonBean();
        personBean.setEmail(email);
        personBean.setMobile(mobile);
        personBean.setName(name);
        personBean.setIdNo(idNo);
        personBean.setPersonArea(LegalAreaType.of(area));
        personBean.setOrgan(organ);
        personBean.setTitle(title);
        personBean.setAddress(address);

        logger.info("creating user account");
        AccountService accountService = AccountServiceFactory.instance();
        AddAccountResult result = accountService.addAccount(personBean);
        if (CODE_SUCCESS != result.getErrCode()) {
            logger.info("create user account failed，errCode={},msg={}", result.getErrCode(), result.getMsg());
        } else {
            logger.info("create user account success！accountId = {}", result.getAccountId());
        }
        return result;

    }

    public static AddAccountResult addPersonAccount(PersonBean bean) {
        logger.info("creating user account");
        AccountService accountService = AccountServiceFactory.instance();
        AddAccountResult result = accountService.addAccount(bean);
        if (CODE_SUCCESS != result.getErrCode()) {
            logger.info("create user account failed，errCode={},msg={}", result.getErrCode(), result.getMsg());
        } else {
            logger.info("create user account success！accountId = {}", result.getAccountId());
        }
        return result;

    }

    public static AddAccountResult addOrganizeAccount(OrganizeBean bean) {
        logger.info("creating user account");
        AccountService accountService = AccountServiceFactory.instance();
        AddAccountResult result = accountService.addAccount(bean);
        if (CODE_SUCCESS != result.getErrCode()) {
            logger.info("create organize account failed，errCode={},msg={}", result.getErrCode(), result.getMsg());
        } else {
            logger.info("create organize account success！accountId = {}", result.getAccountId());
        }
        return result;
    }

    /***
     * 使用模板创建个人账户的印章； 使用到接口：sealService.addTemplateSeal(accountId,
     * PersonTemplateType.SQUARE, SealColor.RED);
     */
    public static AddSealResult addPersonTemplateSeal(String accountId) {
        return addPersonTemplateSeal(accountId, PersonTemplateType.RECTANGLE, SealColor.RED);

    }

    /***
     * 使用模板创建个人账户的印章； 使用到接口：sealService.addTemplateSeal(accountId,
     * PersonTemplateType.SQUARE, SealColor.RED);
     */
    public static AddSealResult addPersonTemplateSeal(String accountId, PersonTemplateType type, SealColor color) {
        logger.info("creating personal template,accountId:{},type:{},color:{}", accountId, type, color);
        SealService sealService = SealServiceFactory.instance();

        AddSealResult result = sealService.addTemplateSeal(accountId, type, color);
        if (CODE_SUCCESS != result.getErrCode()) {
            logger.info("create personal template failed，errCode={},msg={}", result.getErrCode(), result.getMsg());
        } else {
            logger.info("create personal template success！sealData = {}", result.getSealData());
        }
        return result;

    }
    /***
     * 创建企业账户的印章,该企业账户印章是一个相对概念。可以理解成是你们公司的客户企业印章；
     * 使用到接口：sealService.addTemplateSeal(accountId, OrganizeTemplateType.STAR,
     * SealColor.RED, "合同专用章", "下弦文");
     */
    public static AddSealResult addOrganizeTemplateSeal(String accountId, OrganizeTemplateType type, SealColor color,String hText,String qText) {
        logger.info("creating organize template,accountId:{},type:{},color:{},hText:{},qText:{}", accountId, type, color, hText, qText);
        SealService sealService = SealServiceFactory.instance();
        AddSealResult addSealResult = sealService.addTemplateSeal(accountId, type, color, hText,
                qText);
        if (CODE_SUCCESS != addSealResult.getErrCode()) {
            logger.info("create organize template failed，errCode={},msg={}", addSealResult.getErrCode(), addSealResult.getMsg());
        } else {
            logger.info("create organize template success！sealData = {}", addSealResult.getSealData());
        }
        return addSealResult;

    }



    private static SignatureConfig initSignatureConfig() {
        SignatureConfig sCfg = new SignatureConfig();
        // 算法类型，默认HMAC-SHA256
        sCfg.setAlgorithm(AlgorithmType.HMACSHA256); // 可选RSA，但推荐使用HMACSHA256
        // e签宝公钥，可以从开放平台获取。若算法类型为RSA，此项必填
        sCfg.setEsignPublicKey("");
        // 平台私钥，可以从开放平台下载密钥生成工具生成。若算法类型为RSA，此项必填
        sCfg.setPrivateKey("");
        return sCfg;
    }

    private static HttpConnectionConfig initHttpConnectionConfig() {
        HttpConnectionConfig httpConCfg = new HttpConnectionConfig();
        // 协议类型，默认https
        httpConCfg.setHttpType(HttpType.HTTPS);
        // 请求失败重试次数，默认5次
        httpConCfg.setRetry(DEFAULT_RETRY_TIME);
        // 代理服务IP地址
        httpConCfg.setProxyIp(null);
        // 代理服务端口
        httpConCfg.setProxyPort(0);
        return httpConCfg;
    }

    private static HttpConnectionConfig initHttpConnectionConfig(String proxyId, Integer port) {
        HttpConnectionConfig httpConCfg = new HttpConnectionConfig();
        // 协议类型，默认https
        httpConCfg.setHttpType(HttpType.HTTPS);
        // 请求失败重试次数，默认5次
        httpConCfg.setRetry(DEFAULT_RETRY_TIME);
        // 代理服务IP地址
        httpConCfg.setProxyIp(proxyId);
        // 代理服务端口
        httpConCfg.setProxyPort(port);
        return httpConCfg;
    }

    private static ProjectConfig initProjectConfig(String projectId, String projectSecret, String apiUrl) {
        ProjectConfig proCfg = new ProjectConfig();
        proCfg.setProjectId(projectId);
        proCfg.setProjectSecret(projectSecret);
        proCfg.setItsmApiUrl(apiUrl);
        return proCfg;
    }

    private static boolean anyNull(Object... objects) {
        if (objects != null) {
            for (Object o : objects) {
                if (o == null) {
                    return true;
                }
            }
        }
        return false;
    }

    private static boolean anyNullOrBlank(Object... objects) {
        if (objects != null) {
            for (Object o : objects) {
                if (o == null) {
                    return true;
                } else if (o instanceof String) {
                    String s = (String) o;
                    return StringUtils.isBlank(s);
                }
            }
        }
        return false;
    }

}
