/*
 * #{copyright}#
 */
package com.hand.hap.adaptor.impl;

import com.hand.hap.account.dto.Role;
import com.hand.hap.account.dto.User;
import com.hand.hap.account.exception.RoleException;
import com.hand.hap.account.exception.UserException;
import com.hand.hap.account.mapper.RoleMapper;
import com.hand.hap.account.service.IRole;
import com.hand.hap.account.service.IRoleService;
import com.hand.hap.account.service.IUserService;
import com.hand.hap.adaptor.ILoginAdaptor;
import com.hand.hap.core.BaseConstants;
import com.hand.hap.core.IRequest;
import com.hand.hap.core.components.CaptchaConfig;
import com.hand.hap.core.components.SysConfigManager;
import com.hand.hap.core.exception.BaseException;
import com.hand.hap.core.exception.IBaseException;
import com.hand.hap.core.impl.RequestHelper;
import com.hand.hap.core.util.TimeZoneUtil;
import com.hand.hap.security.IUserSecurityStrategy;
import com.hand.hap.security.TokenUtils;
import com.hand.hap.security.captcha.ICaptchaManager;
import com.hand.hap.security.service.impl.UserSecurityStrategyManager;
import com.hand.hap.system.dto.ResponseData;
import com.hand.hls.sys.dto.LeafRole;
import com.hand.hls.sys.mapper.UserExtensionMapper;
import com.hand.hls.sys.service.ILeafRoleService;
import leaf.sys.dto.SysSession;
import leaf.sys.mapper.SysSessionMapper;
import leaf.utils.ConfigUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.ui.ModelMap;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.support.RequestContextUtils;
import org.springframework.web.util.HtmlUtils;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.util.*;

/**
 * 默认登陆代理类.
 *
 * @author njq.niu@hand-china.com
 * @author xiawang.liu@hand-china.com 2016年1月19日 TODO:URL和页面分开
 */
public class DefaultLoginAdaptor implements ILoginAdaptor {

    private static final boolean VALIDATE_CAPTCHA = true;

    // 校验码
    private static final String KEY_VERIFICODE = "verifiCode";

    // 默认主页
    private static final String VIEW_INDEX = "/";

    // 默认的登录页
    private static final String VIEW_LOGIN = "/login";

    // 默认角色选择路径
    private static final String VIEW_ROLE_SELECT = "/role";
    private static final String DEFAULT_HOME_PAGE = BaseConstants.VIEW_REDIRECT + "main.lview";


    @Autowired
    private ICaptchaManager captchaManager;

    @Autowired
    private MessageSource messageSource;

    @Autowired
    @Qualifier("roleServiceImpl")
    private IRoleService roleService;

    @Autowired
    private ILeafRoleService leafRoleService;

    @Autowired
    private UserExtensionMapper userExtensionMapper;

    @Autowired
    private RoleMapper roleMapper;
    @Autowired
    private SysSessionMapper sysSessionMapper;

    @Autowired
    private IUserService userService;

    @Autowired
    private CaptchaConfig captchaConfig;

    @Autowired
    private SysConfigManager sysConfigManager;

    @Autowired
    UserSecurityStrategyManager userSecurityStrategyManager;

    public ModelAndView doLogin(User user, HttpServletRequest request, HttpServletResponse response) {

        ModelAndView view = new ModelAndView();
        Locale locale = RequestContextUtils.getLocale(request);
        view.setViewName(getLoginView(request));
        try {
            beforeLogin(view, user, request, response);
            checkCaptcha(view, user, request, response);
            user = userService.login(user);
            HttpSession session = request.getSession(true);
            session.setAttribute(User.FIELD_USER_ID, user.getUserId());
            session.setAttribute(User.FIELD_USER_NAME, user.getUserName());
            session.setAttribute(IRequest.FIELD_LOCALE, locale.toString());
            setTimeZoneFromPreference(session, user.getUserId());
            generateSecurityKey(session);
            afterLogin(view, user, request, response);
        } catch (UserException e) {
            view.addObject("msg", messageSource.getMessage(e.getCode(), e.getParameters(), locale));
            view.addObject("code", e.getCode());
            processLoginException(view, user, e, request, response);
        }
        return view;
    }

    private void setTimeZoneFromPreference(HttpSession session, Long accountId) {
        String tz = "GMT+0800";
        if (StringUtils.isBlank(tz)) {
            tz = TimeZoneUtil.toGMTFormat(TimeZone.getDefault());
        }
        session.setAttribute(BaseConstants.PREFERENCE_TIME_ZONE, tz);
    }

    private String generateSecurityKey(HttpSession session) {
        return TokenUtils.setSecurityKey(session);
    }

    /**
     * 登陆前逻辑.
     *
     * @param view     视图
     * @param account  账号
     * @param request  请求
     * @param response 响应
     * @throws UserException 异常
     */
    protected void beforeLogin(ModelAndView view, User account, HttpServletRequest request,
                               HttpServletResponse response) throws UserException {

    }

    /**
     * 处理登陆异常.
     *
     * @param view
     * @param account
     * @param e
     * @param request
     * @param response
     */
    protected void processLoginException(ModelAndView view, User account, UserException e, HttpServletRequest request,
                                         HttpServletResponse response) {

    }

    /**
     * 校验验证码是否正确.
     *
     * @param view     视图
     * @param user     账号
     * @param request  请求
     * @param response 响应
     * @throws UserException 异常
     */
    private void checkCaptcha(ModelAndView view, User user, HttpServletRequest request, HttpServletResponse response)
            throws UserException {
        if (VALIDATE_CAPTCHA) {
            Cookie cookie = WebUtils.getCookie(request, captchaManager.getCaptchaKeyName());
            String captchaCode = request.getParameter(KEY_VERIFICODE);
            if (cookie == null || StringUtils.isEmpty(captchaCode)
                    || !captchaManager.checkCaptcha(cookie.getValue(), captchaCode)) {
                // view.addObject("_password", user.getPassword());
                throw new UserException(UserException.ERROR_INVALID_CAPTCHA, UserException.ERROR_INVALID_CAPTCHA, null);
            }
        }
    }

    /**
     * 账号登陆成功后处理逻辑.
     *
     * @param view     视图
     * @param user     账号
     * @param request  请求
     * @param response 响应
     * @throws UserException 异常
     */
    protected void afterLogin(ModelAndView view, User user, HttpServletRequest request, HttpServletResponse response)
            throws UserException {
        view.setViewName(BaseConstants.VIEW_REDIRECT + getRoleView(request));
        Cookie cookie = new Cookie(User.FIELD_USER_NAME, user.getUserName());
        cookie.setPath(StringUtils.defaultIfEmpty(request.getContextPath(), "/"));
        cookie.setMaxAge(-1);
        response.addCookie(cookie);
    }

    @Override
    public ModelAndView doSelectRole(IRole role, HttpServletRequest request, HttpServletResponse response)
            throws RoleException {
        ModelAndView result = new ModelAndView();
        // 选择角色
        HttpSession session = request.getSession(false);
        if (session != null && role != null && role.getRoleId() != null) {
            // leaf 1.0
            if (ConfigUtils.isLeafTargetVersion(ConfigUtils.VERSION_1_0)) {
                session.setAttribute("role_id", role.getRoleId());
                Object session_id = session.getAttribute("session_id");
                if (session_id != null) {
                    SysSession sysSession = new SysSession();
                    sysSession.setSessionId((Long) session_id);
                    sysSession.setRoleId(role.getRoleId());
                    sysSessionMapper.updateByPrimaryKeySelective(sysSession);
                }
            }
            // end leaf 1.0
            Long userId = (Long) session.getAttribute(User.FIELD_USER_ID);
            roleService.checkUserRoleExists(userId, role.getRoleId());
            if (!sysConfigManager.getRoleMergeFlag()) {
                Long[] ids = new Long[1];
                ids[0] = role.getRoleId();
                session.setAttribute(IRequest.FIELD_ALL_ROLE_ID, ids);
            }
            session.setAttribute(IRequest.FIELD_ROLE_ID, role.getRoleId());
            result.setViewName(BaseConstants.VIEW_REDIRECT + getIndexView(request));
        } else {
            result.setViewName(BaseConstants.VIEW_REDIRECT + getLoginView(request));
        }
        return result;
    }

    @Override
    public ModelAndView doSelectRole(Map<String, Object> map, HttpServletRequest request, HttpServletResponse response)
            throws RoleException {
        Long userRoleGroupId = (Long) map.get("user_role_group_id");
        Long roleId = (Long) map.get("role_id");
        Long companyId = (Long) map.get("company_id");
        ModelAndView result = new ModelAndView();
        // 选择角色
        HttpSession session = request.getSession(false);
        if (session != null && userRoleGroupId != null) {
            leafRoleService.checkLeafV1UserRoleExists(userRoleGroupId);
            session.setAttribute("role_id", roleId);
            session.setAttribute("company_id", companyId);
            session.setAttribute("companyId", companyId);
            Object session_id = session.getAttribute("session_id");
            if (session_id != null) {
                SysSession sysSession = new SysSession();
                sysSession.setSessionId((Long) session_id);
                sysSession.setRoleId(roleId);
                sysSession.setCompanyId(companyId);
                sysSessionMapper.updateByPrimaryKeySelective(sysSession);
            }
            Long userId = (Long) session.getAttribute(User.FIELD_USER_ID);

            if (!sysConfigManager.getRoleMergeFlag()) {
                Long[] ids = new Long[1];
                ids[0] = roleId;
                session.setAttribute(IRequest.FIELD_ALL_ROLE_ID, ids);
            }
            session.setAttribute(IRequest.FIELD_ROLE_ID, roleId);
            List<Map<String, String>> maps = sysSessionMapper.selectUserTheme((Long) session.getAttribute(IRequest.FIELD_USER_ID));
            String subject = null;
            if (maps.size() > 0) {
                subject = maps.get(0).get("SUBJECT");
                session.setAttribute("subject", subject);
            }
            if (subject != null) {
                if ("HAP".equalsIgnoreCase(session.getAttribute("subject").toString())) {
                    result.setViewName(BaseConstants.VIEW_REDIRECT + "/main.lview");
                } else {
                    result.setViewName(BaseConstants.VIEW_REDIRECT + "/leaf_main.lview");
                }
            } else {
                result.setViewName(BaseConstants.VIEW_REDIRECT + "/main.lview");
            }

            Object checkAuth = "N", userAuth = "N", roleAuth = "N";
            Map<String, Long> bizMap = userExtensionMapper.selectBizCompanyId(companyId);
            Long bizCompanyId = companyId;
            if (bizMap != null && bizMap.containsKey("bizCompanyId")) {
                bizCompanyId = bizMap.get("bizCompanyId");
            }
            Map<String, String> checkAuthMap = userExtensionMapper.selectCheckAuth(bizCompanyId);
            Map<String, String> authMap = userExtensionMapper.selectUserRoleAuth(bizCompanyId);
            checkAuth = getDefaultValue(checkAuthMap, "checkAuthority", "N");
            userAuth = getDefaultValue(authMap, "userAuthority", "N");
            roleAuth = getDefaultValue(authMap, "roleAuthority", "N");

            session.setAttribute("top_biz_company_id", bizCompanyId);
            session.setAttribute("owner_user_aut_flag", checkAuth);
            session.setAttribute("enable_user_authority", userAuth);
            session.setAttribute("enable_role_authority", roleAuth);
        } else {
            result.setViewName(BaseConstants.VIEW_REDIRECT + getLoginView(request));
        }
        return result;
    }


    private Object getDefaultValue(Map map, String key, Object defaultValue) {
        if (map == null) {
            return defaultValue;
        }
        Object value = map.get(key);
        if (value != null && StringUtils.isNotEmpty(value.toString())) {
            return value;
        }
        return defaultValue;
    }

    /**
     * 获取主界面.
     *
     * @param request
     * @return 视图
     */
    protected String getIndexView(HttpServletRequest request) {
        return VIEW_INDEX;
    }

    /**
     * 获取登陆界面.
     *
     * @param request
     * @return 视图
     */
    protected String getLoginView(HttpServletRequest request) {
        return VIEW_LOGIN;
    }


    /**
     * 获取角色选择界面.
     *
     * @param request
     * @return 视图
     */
    protected String getRoleView(HttpServletRequest request) {
        return VIEW_ROLE_SELECT;
    }

    /**
     * 集成类中可扩展此方法实现不同的userService.
     *
     * @return IUserService
     */
    public IUserService getUserService() {
        return userService;
    }

    @Override
    public ModelAndView indexView(HttpServletRequest request, HttpServletResponse response) {
        HttpSession session = request.getSession(false);
        ModelAndView mav = indexModelAndView(request, response);
        if (session != null) {
            // 获取user
            Long userId = (Long) session.getAttribute(User.FIELD_USER_ID);
            if (userId != null) {
                if (session.getAttribute(User.LOGIN_CHANGE_INDEX) != null) {
                    User user = new User();
                    user.setUserId(userId);
                    user = userService.selectByPrimaryKey(RequestHelper.createServiceRequest(request), user);
                    List<IUserSecurityStrategy> userSecurityStrategies = userSecurityStrategyManager.getUserSecurityStrategyList();
                    for (IUserSecurityStrategy userSecurityStrategy : userSecurityStrategies) {
                        ModelAndView mv = userSecurityStrategy.loginVerifyStrategy(user, request);
                        if (mv != null) {
                            return mv;
                        }
                    }
                    session.removeAttribute(User.LOGIN_CHANGE_INDEX);
                }
            } else {
                return new ModelAndView(BaseConstants.VIEW_REDIRECT + getLoginView(request));
            }
            // 角色选择
            if (!sysConfigManager.getRoleMergeFlag()) {
                Long roleId = (Long) session.getAttribute(Role.FIELD_ROLE_ID);
                if (roleId == null) {
                    return new ModelAndView(BaseConstants.VIEW_REDIRECT + getRoleView(request));
                }
                //查询用户角色为角色切换。
                User user = new User();
                user.setUserId(userId);
                List<IRole> roles = roleService.selectRolesByUser(RequestHelper.createServiceRequest(request), user);

                mav.addObject("SYS_USER_ROLES", roles);
                mav.addObject("CURRENT_USER_ROLE", roleId);
            }
        }

        String sysTitle = sysConfigManager.getSysTitle();
        mav.addObject("SYS_TITLE", sysTitle);
        if (mav.getViewName().equals(DEFAULT_HOME_PAGE)) {
            mav.getModelMap().clear();
        }
        return mav;
    }

    /**
     * 默认登陆页面.
     *
     * @param request
     * @param response
     * @return 视图
     */
    public ModelAndView indexModelAndView(HttpServletRequest request, HttpServletResponse response) {
        return new ModelAndView(DEFAULT_HOME_PAGE);
    }

    @Override
    public ModelAndView loginView(HttpServletRequest request, HttpServletResponse response) {
        HttpSession session = request.getSession(false);
        if(session != null){
//            session.getAttribute()
            if(null != session.getAttribute(Role.FIELD_ROLE_ID)){
                // role selected
                return indexView(request, response);
            }
            if(null != session.getAttribute(User.FIELD_USER_ID)){
//                user login
                try {
                    return roleView(request, response);
                } catch (RoleException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        ModelAndView view = new ModelAndView(getLoginView(request));
        // 配置3次以后开启验证码
        Cookie cookie = WebUtils.getCookie(request, CaptchaConfig.LOGIN_KEY);
        if (captchaConfig.getWrongTimes() > 0) {
            if (cookie == null) {
                String uuid = UUID.randomUUID().toString();
                cookie = new Cookie(CaptchaConfig.LOGIN_KEY, uuid);
                cookie.setPath(StringUtils.defaultIfEmpty(request.getContextPath(), "/"));
                cookie.setMaxAge(captchaConfig.getExpire());
                response.addCookie(cookie);
                captchaConfig.updateLoginFailureInfo(cookie);
            }
        }

        // 向前端传递是否开启验证码
        view.addObject("ENABLE_CAPTCHA", captchaConfig.isEnableCaptcha(cookie));

        view.addObject("SYS_TITLE", sysConfigManager.getSysTitle());

        Boolean error = (Boolean) request.getAttribute("error");
        Throwable exception = (Exception) request.getAttribute("exception");

        while (exception != null && exception.getCause() != null) {
            exception = exception.getCause();
        }
        String code = UserException.ERROR_USER_PASSWORD;
        if (exception instanceof BaseException) {
            code = ((BaseException) exception).getDescriptionKey();
        }

        if (error != null && error) {
            String msg;
            Locale locale = RequestContextUtils.getLocale(request);
            msg = messageSource.getMessage(code, null, locale);
            view.addObject("msg", msg);
        }
        return view;
    }

    @Override
    public ModelAndView roleView(HttpServletRequest request, HttpServletResponse response) throws RoleException {
        ModelAndView mv = new ModelAndView(getRoleView(request));
        HttpSession session = request.getSession(false);
        mv.addObject("SYS_TITLE", sysConfigManager.getSysTitle());
        if (session != null) {
            // 获取user
            Long userId = (Long) session.getAttribute(User.FIELD_USER_ID);
            if (userId != null) {
                User user = new User();
                user.setUserId(userId);
                session.setAttribute(User.FIELD_USER_ID, userId);
                addCookie(User.FIELD_USER_ID, userId.toString(), request, response);


                if (ConfigUtils.isLeafTargetVersion(ConfigUtils.VERSION_1_0)) {
                    // leaf 1.0 version, get roles from different table
                    List<LeafRole> leafRoles = userExtensionMapper.selectLeafV1UserRoles(user.getUserId());
                    if(leafRoles.size()==1&&ConfigUtils.isNotJumpRole()){
                        HashMap<String, Object> map = new HashMap<>();
                        LeafRole leafRole = leafRoles.get(0);
                        map.put("user_role_group_id", leafRole.getUserRoleGroupId());
                        map.put("role_id", leafRole.getRoleId());
                        map.put("company_id", leafRole.getCompanyId());
                        return doSelectRole(map,request, response);
                    }
                    mv.addObject("roles", leafRoles);
                } else {
                    List<IRole> roles = roleService.selectRolesByUser(RequestHelper.createServiceRequest(request), user);
                    mv.addObject("roles", roles);
                }
            }
        }
        return mv;
    }

    private void addCookie(String cookieName, String cookieValue, HttpServletRequest request,
                           HttpServletResponse response) {
        Cookie cookie = new Cookie(cookieName, cookieValue);
        cookie.setPath(StringUtils.defaultIfEmpty(request.getContextPath(), "/"));
        cookie.setMaxAge(-1);
        response.addCookie(cookie);
    }

    @Override
    public ResponseData sessionExpiredLogin(User account, HttpServletRequest request, HttpServletResponse response)
            throws RoleException {
        ResponseData data = new ResponseData();
        ModelAndView view = this.doLogin(account, request, response);
        ModelMap mm = view.getModelMap();
        if (mm.containsAttribute("code")) {
            data.setSuccess(false);
            data.setCode((String) mm.get("code"));
            data.setMessage((String) mm.get("msg"));
        } else {
            Object userIdObj = request.getParameter(User.FIELD_USER_ID);
            Object roleIdObj = request.getParameter(IRequest.FIELD_ROLE_ID);
            if (userIdObj != null && roleIdObj != null) {
                Long userId = Long.valueOf(userIdObj.toString()), roleId = Long.valueOf(roleIdObj.toString());
                roleService.checkUserRoleExists(userId, roleId);
                HttpSession session = request.getSession();
                session.setAttribute(User.FIELD_USER_ID, userId);
                session.setAttribute(IRequest.FIELD_ROLE_ID, roleId);
            }
        }
        return data;
    }

    @Override
    public ModelAndView defaultView(HttpServletRequest request, HttpServletResponse response) {
        return new ModelAndView(BaseConstants.VIEW_REDIRECT + sysConfigManager.getDefaultTargetUrl());
    }
}
