# vuex store 使用指南

* [vuex 官方网址](https://vuex.vuejs.org/zh-cn/)

store 配置是按模块区分的，所以添加配置时同一个模块的 store 放在一个文件中添加新的

store 配置很简单，步骤如下：

1.  在 `store/modules` 目录下添加模块对应的 store 文件，内容格式如下：

    ```js
    const state = {
      increment: 0,
    }

    const getters = {}
    Object.keys(state).forEach(prop => {
      getters[prop] = state => state[prop]
    })

    const mutations = {}
    Object.keys(state).forEach(prop => {
      const setProp = `set${prop.charAt(0).toUpperCase()}${prop.slice(1)}`
      mutations[setProp] = (state, payload) => {
        state[prop] = payload
      }
    })

    // actions不做统一处理，因为大多数情况下，并不需要使用异步设置值
    // 若需要使用actions，注意：mutation_type的命名规则为set开头加上state对象属性的camel形式
    // const actions = {
    //   async setIncrement({ commit }, value) {
    //     commit('setIncrement', await value);
    //   },
    // };
    const actions = {}

    export default {
      state,
      getters,
      mutations,
      actions,
    }
    ```

    要添加缓存数据时，只需要在 state 对象中添加属性即可，getters 和 mutations 对象通常不需要修改

    由于根据以往项目经验，很少会用到异步修改值的功能，所以 actions 没做统一处理，若需要自行拓展，按以上注释掉的部分修改即可

2.  在 `store/index.js` 文件中引入添加的 store 文件，并在 modules 字段中注入，如下：

    ```js
    import helloWorld from './modules/helloWorld'

    export default new Vuex.Store({
      modules: {
        helloWorld,
      },
      state: {},
      getters: {},
      mutations: {},
      actions: {},
    })
    ```

    如果有些 store 属性是全局的，不属于某个模块，可以直接在 `store/index.js` 如上几个空对象中添加

## 组件中使用 store 属性

所有的 state 中的属性都会在 getters 中生成同名的属性，以及在 mutations 中生成以 `set + 首字母大写的 state 属性` 的属性，所以用法如下：

```js
import {
  mapGetters,
  mapMutations,
  // mapActions,
} from 'vuex'

export default {
  computed: {
    ...mapGetters(['increment']),
  },
  methods: {
    ...mapMutations(['setIncrement']),
    // ...mapActions(['setIncrement']),
    click() {
      this.setIncrement(this.increment + 1)
    },
  },
}
```
