# http 使用指南

## 说明

http 为基于 axios 二次封装，使用方式保持与 axios 一致。
[axios 官方教程](https://github.com/axios/axios)

## 使用指南

1.  在 src/api 文件夹下定义你的业务相关接口函数并导出

```js
// api/home.js中
import http from '@/utils/http'

// 注意：http 中会自动给 url 前拼接环境定义的 baseUrl。定义的 url 中无需包含 baseUrl
// 如需修改环境配置参数（如：baseUrl），可在 src/data/config 下修改
// http 中默认会携带 localStorage.token（若采用hmap login方法，localStorage.token 会在登录成功后写入），无需手动设置
// 仅仅放心的去使用它 ：）
export function getSomething() {
  const url = '/fetch'
  return http.get(url)
}

export function postSomething(data) {
  const url = '/fetch2'
  return http.post(url, data)
}
```

2.  在你的 src/views/ 页面中导入你需要用到的接口函数，使用它

```js
// views/home.vue中
import { getSomething, postSomething } from 'api/home'

export default {
  /* ... */
  methods: {
    async handle() {
      // 使用 async/await 写法时方法前要加 async
      // 使用 async/await 写法（推荐）
      try {
        const data = {} // post参数
        const res = await postSomething(data) // res即为response.data
      } catch (e) {
        // 处理error
      }

      // 使用 promise 写法
      getSomething()
        .then(res => {
          // 注意：res即为response.data，不需要进行response.data.success === true的判断
          // 若项目接口无 response.data.success 字段，修改 src/utils/http/interceptors.js 第13行
        })
        .catch(err => {
          // 所有error情况统一在此处理，包括返回状态码为200，和 response.data.success 为 false 的情况。http 内部默认会对一些错误码做提示（alert）处理
        })
    },
  },
  /* ... */
}
```

## 不统一处理接口返回的 success 字段

如果项目接口规范不到位，或者因为一些原因无法做到所有接口返回中都有 success 字段，那么可能无法对 success 字段做统一处理，需要修改下 src/utils/http/interceptors.js 文件：

  ```js
  // 请求返回之后的预处理函数：若返回状态码为200，但实际请求未完成，将异常处理统一到错误处理
  const responseAfter = response => response.data.success ? response.data : Promise.reject(response)
  // 若项目接口无 response.data.success 状态字段，注释上一行，使用下行
  // const responseAfter = response => response
  ```

改为

  ```js
  // 请求返回之后的预处理函数：若返回状态码为200，但实际请求未完成，将异常处理统一到错误处理
  // const responseAfter = response => response.data.success ? response.data : Promise.reject(response)
  // 若项目接口无 response.data.success 状态字段，注释上一行，使用下行
  const responseAfter = response => response
  ```

然后自行对每个接口单独做处理。
