/*
 * @Author: zong.wang01@hand-china.com
 * @Date: 2024-07-30 21:21:37
 * @LastEditors: zong.wang01@hand-china.com
 * @LastEditTime: 2024-09-19 16:05:29
 * @Version: 1.0.0
 * @Description: 
 * @Copyright: Copyright (c) 2021, Hand-RongJing
 */
import Vue from 'vue';
import {compact, head, isFunction, isNil, isNumber, isUndefined, upperFirst} from 'lodash';
import {getCustomizedProp, getHeaderExtrasProp, getCurrentUser} from '../utils/utils';

/**
 * 格式化代码
 * @param  code {string}
 * @returns {string} 格式化代码
 */
const parseCode = code => {
  return code ? code.toLowerCase().replace(/_(\w)/g, (all, letter) => letter.toUpperCase()) : '';
};

/**
 * 基本建造类
 */
class BaseBuilder {
  init() {
    Object.keys(this).forEach(key => {
      const withName = `with${upperFirst(key)}`;
      this[withName] = value => {
        this[key] = value;
        return this;
      };
    });
  }
}

/**
 * 列信息建造者
 * @class
 * @constructor
 * @public
 */
export class ColumnBuilder extends BaseBuilder {
  constructor() {
    super();
    /**
     * 列信息
     * @type {Array.<{columnSeq: number, rowSeq: number, columnName: string, cascadeValue: string, validationTypeDisplay: string, columnCount: number, rowCount: number, inputModeDisplay: string, width: number}>}
     * @property
     * @public
     */
    this.fields = [];
    /**
     * 类型
     * @type {string} 组件类型
     * @property
     * @public
     */
    this.type = '';

    /**
     * 组件code
     * @type {string}
     */
    this.tabCode = '';

    /**
     * 父级组件code
     * @type {string}
     */
    this.parentTabCode = '';

    /**
     * componentRenderers
     * @type {Array}
     */
    this.componentRenderers = [];

    /**
     * layoutCode
     * @type {string}
     */
    this.layoutCode = '';

    /**
     * 页脚渲染
     * @type {*[]}
     */
    this.tableFooterRenderers = [];

    this.createElement= () => {}
    super.init();
  }

  getCustomizedRenderer(columnName) {
    const currentRenderers = this.getCurrentRenderers(columnName, this.componentRenderers);
    let renderer = currentRenderers.length > 0 ? currentRenderers[0].renderer : undefined;
    let formRenderer = currentRenderers.length > 0 ? currentRenderers[0].formRenderer : undefined;
    let className = currentRenderers.length > 0 ? currentRenderers[0].className : undefined;
    return { renderer, formRenderer, className };
  }

  getCurrentRenderers(columnName, renderers) {
    return getCustomizedProp(this.layoutCode, this.tabCode, columnName, renderers);
  }

  /**
   * 生成表单下字段组件数组
   * @param {Array.<{columnName: string, columnSeq: number, rowSeq: number, cascadeValue: string, validationTypeDisplay: string, columnCount: number, rowCount: number, inputModeDisplay: string, width: number}>} fields
   * @param {string} cascadeKey
   * @return
   */
  initFields(fields, cascadeKey = '') {
    const newFields = fields.map(field => {
      const customerRender = this.getCustomizedRenderer(field.columnName);
      return {
        ...field,
        readOnly: field.inputModeDisplay === "READONLY",
        required: field.inputModeDisplay === "REQUIRED",
        clearFlag: field.clearButton === 'Y',
        renderer: customerRender.renderer,
        dataClass: customerRender.className
      }
      // if (field.validationTypeDisplay === 'UrlField') {
      //   const {renderer} = this.getCustomizedRenderer(field.columnName, field);
      //   // if (renderer instanceof FormField) {
      //   return renderer || <div label={field.description}>{field.columnName}</div>;
      //   /* } else {
      //     return renderer || <div label={field.description} newLine={field.columnSeq === 1}>{field.columnName}</div>;
      //   } */
      // }
      // const dataClass= `${this.layoutCode}-${this.tabCode}-${field.columnName}`.toLowerCase();
      // return FormField({
      //   ...field,
      //   key: field.id,
      //   cascadeKey,
      //   dataSet: this.dataSet,
      //   'data-class':dataClass,
      //   // showTitle: true,
      // });
    });
    return {
      fields: newFields.filter(field => isNumber(field.columnSeq) && isNumber(field.rowSeq)),
      _originFields: newFields,
    };
  }

  /**
   * 处理表单字段
   * @returns {{columnName: string, cascadeValue: string}|*}
   */
  parseFormFields() {
    let cascadeKey = '';
    const formFields = this.fields.reduce((acc, cur) => {
      const {cascadeValue, formatCascadeFlag} = cur;
      const result = {...acc};
      if (formatCascadeFlag === 'Y') {
        cascadeKey = cur.columnName;
      }
      if (result[cascadeValue]) {
        result[cascadeValue].push(cur);
      } else {
        result[cascadeValue] = [cur];
      }
      return result;
    }, {});

    if (!cascadeKey) {
      return { DEFAULT: this.initFields(this.fields) };
    } else {
      const formComponents = {};
      Object.keys(formFields).forEach(cascadeValue => {
        formComponents[cascadeValue] = this.initFields(formFields[cascadeValue], cascadeKey);
      });
      return formComponents;
    }
  }

  build() {
    switch (this.type) {
      case 'Table':
      case 'Tree': {
        const newFields = this.fields.map(field => { // .filter(o => o.isVisible === 'Y')
          const customizedRenderer = this.getCustomizedRenderer(field.columnName);
          return {
            ...field,
            readOnly: field.inputModeDisplay === "READONLY",
            required: field.inputModeDisplay === "REQUIRED",
            clearFlag: field.clearButton === 'Y',
            renderer: customizedRenderer.renderer,
            formRenderer: customizedRenderer.formRenderer,
            dataClass: customizedRenderer.className
          }
        });
        // return newFields;
        return {
          fields: newFields.filter(o => o.isVisible === 'Y'),
          _originFields: newFields,
        }
      }
      case 'Form':
        return this.parseFormFields();
      default:
    }
  }
}

/**
 * 表格、表单按钮建造类
 */
// export class TabButtonBuilder extends BaseBuilder {
//   constructor() {
//     super();

//     /**
//      * 组件级按钮
//      * @type {Array.<{name: string, description: string}>}
//      */
//     this.tabButtons = [];

//     /**
//      *
//      * @type {Array.<{tabCode: string, layoutCode: string, name: string, clickFunction: Function}>}
//      */
//     this.componentButtons = [];

//     /**
//      * 组件code
//      * @type {string}
//      */
//     this.tabCode = '';

//     this.layoutCode = '';
//     /**
//      * 组件类型
//      * @type {string}
//      */
//     this.type = '';

//     /**
//      * 是否只读
//      * @type {boolean}
//      */
//     this.readOnly = false;

//     this.setBtnRef = null;

//     this.createElement = () => {};

//     this.init();
//   }

//   /**
//    * 获取点击事件
//    * @param {string} name - 名称
//    * @returns {Function} 点击事件
//    */
//   getClickFunction(name) {
//     if (this.type === 'Form') {
//       switch (name) {
//         case 'query':
//           console.log('表单查询')
//           break;
//         case 'reset':
//           console.log('表单重置')
//           break;
//         default:
          
//       }
//     } else if (this.type === 'Table') {
//       switch (name) {
//         case 'create':
//           console.log('创建')
//           break;
//         case 'query':
//           console.log('查询')
//           break;
//         case 'export':
//           console.log('导出')
//           break;
//         case 'remove':
//           console.log('删除')
//           break;
//         case 'save':
//           console.log('保存')
//           break; 
//         default:
          
//       }
//     }
    
//   }

//   build() {
//     return this.tabButtons.map(btn => {
//       return this.createElement(
//         'h-button', // 元素类型
//         { 		 // 属性对象
//           props: {
//             type: "primary",
//             key: btn.name
//           },
//           // on: {
//           //   click: this.test,
//           //   test2: function(){
//           //     console.log('22222')
//           //   }
//           // },
//           nativeOn: {
//             click: () => this.getClickFunction(btn.name)
//           }
//         },
//         [ // 子元素数组
//           btn.description, // 文本内容
//         ]
//       )
//     })
//   }
// }

class BaseComponentBuilder extends BaseBuilder {
  constructor() {
    super();
    /**
     * 数据集
     * @type {undefined|DataSet}
     * @public
     */
    this.dataSet = undefined;
    /**
     * 组件类型
     * @type {undefined|string}
     * @public
     */
    this.tabType = undefined;
    /**
     * 列
     * @type {undefined|React.ComponentType[]}
     * @public
     */
    this.columns = undefined;
    /**
     * 关联布局字段
     * @type {string|undefined}
     */
    this.cascadeKey = undefined;
    /**
     * 按钮
     * @type {Array.<React.ReactElement>}
     */
    this.buttons = [];
    this.props = {};

    /**
     * componentHeaderExtras
     * @type {Array}
     */
    this.componentHeaderExtras = [];

    this.staticComponents = [];

    this.componentButtons = [];

    this.setTabRef = null;

    this.routeParams = {}; // 路由信息

    this.createElement = () => {};

    this.parentProps = {};

    super.init();
  }

  getHeader(props) {
    const {renderer} = this.getHeaderRenderer();
    if (props.description || renderer) {
      return (
        <fragment>
          {props.description}
          {renderer && renderer()}
        </fragment>
      )
    }
  }

  getHeaderRenderer() {
    const currentHeaderExtra = getHeaderExtrasProp(this.layoutCode, this.tabCode, this.componentHeaderExtras);
    let renderer = currentHeaderExtra.length > 0 ? currentHeaderExtra[0].renderer : undefined;
    if (renderer) {
      renderer = renderer({
        dataSet: this.dataSet,
        record: this.dataSet ? this.dataSet.current : undefined,
      });
    }
    return {renderer};
  }


  /**
   * table props
   * @returns {{}}
   */
  loadTableProps(props) {
    if (props.rowHeight) {
      return {
        rowHeight: props.rowHeight,
      };
    } else {
      return {};
    }
  }

  /**
   * form props
   * @returns {{}}
   */
  
  loadFormProps() {
    return {};
  }

  /**
   *
   * @param columns {Array.<DetailedReactHTMLElement>}
   * @returns  {Array.<DetailedReactHTMLElement>}
   */
  
  loadTableColumns(columns) {
    return columns;
  }

  /**
   * @param columns {Array.<DetailedReactHTMLElement>}
   * @returns  {Array.<DetailedReactHTMLElement>}
   */
  
  loadFormColumns(columns) {
    return columns;
  }

  urlHandler(url) {
    const allInfo = { ...getCurrentUser(), ...this.routeParams };
    if (url === undefined || url === null) {
      return url;
    }
    const regex = /\{(.+?)\}/g;
    const str1 = url.match(regex); // 包含大括号
    if (str1 === null) {
      return url;
    }
    str1.forEach(item => {
      const str2 = item.substr(1, item.length - 2); // 去除大括号
      // eslint-disable-next-line no-param-reassign
      url = url.replace(item, allInfo[str2]);
    });
    return url;
  }

  handleUrls() {
    const {readUrl, createUrl, updateUrl, destroyUrl} = this.props;
    this.props._originUrl = {
      readUrl,
      createUrl,
      updateUrl,
      destroyUrl
    }
    this.props.readUrl = this.urlHandler(readUrl);
    this.props.createUrl = this.urlHandler(createUrl);
    this.props.updateUrl = this.urlHandler(updateUrl);
    this.props.destroyUrl = this.urlHandler(destroyUrl);
  }

  build() {
    this.handleUrls();
    const refKey = `${parseCode(this.layoutCode)}_${parseCode(this.tabCode)}_${this.tabType.toLowerCase()}Ref`;
    switch (this.tabType) {
      case 'Table':
      case 'Tree':
        return this.createElement(
          'd-table',
          {
            props: {
              key: refKey,
              tabInfo: this.props,
              // buttons: this.buttons,
              columns: this.columns.fields,
              originColumns: this.columns._originFields,
              componentButtons: this.componentButtons
            },
            ref:`DTABLE_${this.tabCode}`,
          },
          []
        );
      case 'Form':
        // 不存在baseTable意味着是查询组件 只有查询组件有按钮
        const { cascadeKey } = this;
        const newFields = cascadeKey ? this.columns[cascadeKey] || this.columns.DEFAULT : this.columns.DEFAULT;
        if (this.props.baseTable) { // 正常表单
          return this.createElement(
            'd-form',
            {
              props: {
                formType: 'normal',
                showTitle: true,
                fields: newFields.fields,
                originFields: newFields._originFields,
                tabInfo: this.props,
                componentButtons: this.componentButtons
              },
              ref:`DFORM_${this.tabCode}`,
            }
          );
        } else { // 查询组件
          return this.createElement(
            'query-form',
            {
              props: {
                fields: newFields.fields,
                originFields: newFields._originFields,
                // buttons: this.props.tabButtons,
                tabInfo: this.props,
                componentButtons: this.componentButtons
              }
            }
          );
        }
      // 处理嵌入布局
      case 'Layout':
        const components = this.staticComponents.filter(item => item.tabCode === this.tabCode);
        if (components && components.length) {
          const StaticComponent = components[0].component;
          return StaticComponent();
        }
        return null;
       
      default:
        throw new Error('Unsupported tab type');
    }
  }
}

/**
 * 组件建造者
 * @class
 * @constructor
 * @public
 */
export class ComponentBuilder extends BaseComponentBuilder {
  constructor() {
    super();
    /**
     * 布局代码
     * @type {string}
     */
    this.layoutCode = '';
    this.tabCode = '';
    this.difference = undefined;
    /**
     * 历史dataSet
     * @type {DataSet}
     */
    this.historyDataSet = undefined;
    this.historyButtonName = undefined;
    this.currentButtonName = undefined;

    /**
     * componentHeaderExtras
     * @type {Array}
     */
    this.componentHeaderExtras = [];

    /**
     * 数据集
     * @type {undefined|DataSet}
     * @public
     */
    this.dataSet = undefined;
    super.init();
  }


  wrapTab(tab, tabType, cols) {
    return super.wrapTab(tab, tabType, cols);
  }
}