import Vue from 'vue'
import { ToastPlugin, LoadingPlugin, DatetimePlugin } from 'vux'

import { ActionSheetPlugin, NotifyPlugin, NumberKeyboardPlugin, SelectPlugin, ShowPicturePlugin } from '../../index'

import confirm from '../../components/Confirm/index'

Vue.use(ToastPlugin)
Vue.use(LoadingPlugin)
Vue.use(DatetimePlugin)

// Vue.prototype.HlsModal = window.HlsModal = HlsModal

/**
 * 锁屏函数 超过10s后自动解屏用于防止屏幕锁死
 * 自动截屏成弹出错误提示框
 * @param content 锁屏内容
 * @param timeout
 */
export function showLoading (content = '', timeout = 10000) {
  Vue.$vux.loading.show({
    text: content || 'Loading',
  })
  // 10s后自动解屏用于防止屏幕锁死
  setTimeout(() => {
    Vue.$vux.loading.hide()
  }, timeout)
}

/**
 * 隐藏
 */
export function hideLoading () {
  Vue.$vux.loading.hide()
}

/**
 * 长时间顶部提示toast
 * @param content
 * @param time
 */
export function showLongTop (content = '', time = 2000) {
  let text = content || '操作失败'

  Vue.$vux.toast.show({
    text: text,
    type: 'text',
    time: time,
    isShowMask: true,
    position: 'top',
  })
}

/**
 * 长时间中部提示toast
 * @param content
 * @param time
 */
export function showLongCenter (content = '', time = 2000) {
  let text = content || '操作失败'

  Vue.$vux.toast.show({
    text: text,
    type: 'text',
    time: time,
    isShowMask: true,
    position: 'middle',
  })
}

/**
 * 长时间中部提示toast
 * @param content
 * @param time
 */
export function showLongBottom (content = '', time = 2000) {
  let text = content || '操作失败'

  Vue.$vux.toast.show({
    text: text,
    time: time,
    type: 'text',
    isShowMask: true,
    position: 'bottom',
  })
}

/**
 * 成功提示框
 * @param content
 * @param time
 */
export function showSuccess (content = '', time = 2000) {
  Vue.$vux.toast.show({
    text: content || '操作成功',
    time: time,
    isShowMask: true,
    type: 'success',
    position: 'middle',
  })
}

/**
 * 成功提示框
 * @param content
 * @param time
 */
export function showError (content = '', time = 2000) {
  Vue.$vux.toast.show({
    text: content || '操作失败',
    type: 'warn',
    isShowMask: true,
    time: time,
    position: 'middle',
  })
}

/**
 * 弹出是否确认的窗口
 * @param confirmObject.title 标题
 * @param confirmObject.content 内容
 * @param confirmObject.onConfirm 确定函数
 */
export function showConfirm (confirmObject) {
  let def = {
    title: confirmObject.title || '提示',
    content: confirmObject.content || '',
    confirmText: '确定',
    cancelText: '取消',
    onConfirm: () => {
      confirmObject.onConfirm(1)
    },
    onCancel: () => {
      confirmObject.onConfirm(0)
    },
  }
  confirm.confirmShow(def)
}

/**
 * 弹出是否确认的窗口
 * @param confirmObject.title 标题
 * @param confirmObject.content 内容
 * @param confirmObject.onConfirm 确定函数
 */
export function showPopup (confirmObject) {
  let def = {
    title: confirmObject.title || '提示',
    content: confirmObject.content || '',
    confirmText: '确定',
    showCancelButton: false,
    onConfirm: () => {
      confirmObject.onConfirm()
    },
  }
  confirm.confirmShow(def)
}

/**
 * @param actionObject.titleText 弹出框的标题可空
 * @param actionObject.callback 点击按钮的回调函数 回传buttonArray数组下标
 * @param actionObject.buttonArray 按钮数组支持[string,string],[object,object],
 *        当为后一种是 object为{text:'拍照',type:'primary'},type支持 primary,warn,disabled
 *
 *  {
 *      titleText: '照片',
 *      buttonArray: [{text:'拍照',type:'warn'}, {text:'从相册取',type:'primary'}],
 *      callback: (index) => {
 *        alert(index);
 *      }
 *   }
 *
 *    {
 *      buttonArray: ['拍照','从相册取'],
 *      callback: (index) => {
 *        alert(index);
 *      }
 *   }
 *
 */
export function showActionSheet (actionObject) {
  if (typeof actionObject === 'object') {
    let buttons = []
    for (let i = 0; i < actionObject.buttonArray.length; i++) {
      if (typeof actionObject.buttonArray[i] === 'object') {
        buttons.push({
          text: actionObject.buttonArray[i].text,
          type: actionObject.buttonArray[i].type,
          callback: actionObject.callback,
        })
      } else {
        buttons.push({
          text: actionObject.buttonArray[i],
          callback: actionObject.callback,
        })
      }
    }
    ActionSheetPlugin.show({
      title: actionObject.titleText || '',
      buttons: buttons,
    })
  }
}

/**
 * 时间选择函数
 * @param timeObject.nowDate 当前展示的时间 可不填
 * @param timeObject.format 时间格式支持不支持秒
 * @param timeObject.callback 点击确定的回调函数
 *
 */
export function showTime (timeObject) {
  let date = new Date().format('yyyy-MM-dd')
  let format = 'YYYY-MM-DD'
  if (timeObject.nowDate) {
    date = timeObject.nowDate
  }
  if (timeObject.format) {
    format = timeObject.format || 'YYYY-MM-DD'
  }
  Vue.$vux.datetime.show({
    cancelText: '取消',
    confirmText: '确定',
    minYear: '1900',
    maxYear: '2200',
    format: format,
    value: date,
    onConfirm (val) {
      timeObject.callback(val)
    },
  })
}

/**
 * 图片放大预览
 * @param imgObject.imgUrl
 */
export function showBigPicture (imgObject) {
  if (typeof imgObject === 'object') {
    ShowPicturePlugin.show({
      imgUrl: imgObject.imgUrl,
      imgOriginalUrl: imgObject.imgOriginalUrl,
      imgSize: imgObject.imgSize,
      width: imgObject.width,
      imgList: imgObject.imgList || [],
      startPosition: imgObject.startPosition,
      loadSuccess: imgObject.loadSuccess,
    })
  }
}

/**
 * 下拉框 支持级联操作 需指定 parent 属性
 * @param selectOption.list Array [{"code": "NP","code_name": "个人"}]
 * @param selectOption.code String "bp_type"
 * @param selectOption.object Object 当前数据对象
 * @param selectOption.returnItem function 回调函数返回index与object
 * var bp_class_list = [
 *    {
 *      "code": "NP",
 *      "code_name": "个人"
 *    },{
 *       "code": "NP1",
 *       "code_name": "个人1"
 *     }];
 *  hlsPopup.selectList({
 *     list: bp_class_list,
 *     code: 'bp_type',
 *     object: {},
 *     returnItem: function (index, obj) {
 *        console.log(obj)
 *      }
 *   })
 *
 */
export function selectList (selectOption) {
  if (typeof selectOption === 'object') {
    let list = []
    let length = selectOption.list.length
    selectOption.list.forEach(function (date, index, array) {
      list.push({
        value: date.code,
        name: date.code_name,
        parent: date.parent,
      })
      if (index === (length - 1)) {
        SelectPlugin.show({
          list: list,
          callBack: selectOption.returnItem,
          code: selectOption.code,
          object: selectOption.object,
          multiple: selectOption.multiple,
        })
      }
    })
  }
}

/**
 * 弹出数字键盘
 * @param keyboardObject.title 键盘的title
 * @param keyboardObject.closeButtonText 键盘的关闭按钮文字
 * @param keyboardObject.keyDown 普通按键按下去事件
 * @param keyboardObject.keyDelete 删除按键按下去事件
 */
export function showNumberKeyboard (keyboardObject) {
  if (typeof keyboardObject === 'object') {
    NumberKeyboardPlugin.show({
      title: keyboardObject.title,
      closeButtonText: keyboardObject.closeButtonText,
      extraKey: keyboardObject.extraKey || '.',
      keyDown: function (text) {
        keyboardObject.keyDown(text)
      },
      keyDelete: function () {
        keyboardObject.keyDelete()
        // console.log('delete')
      },
    })
  }
}

/**
 * 弹出Notify
 * @param notifyObject.content 内容
 * @param notifyObject.position 位置
 * @param notifyObject.time 显示时长
 * @param notifyObject.type 类型  success warning default
 */
export function showNotify (notifyObject) {
  if (typeof notifyObject === 'object') {
    NotifyPlugin.show({
      show: true,
      content: notifyObject.content,
      position: notifyObject.position || 'top',
      time: notifyObject.time || 3000,
      type: notifyObject.type || 'default',
    })
  }
}
