/*
 * @Description: In User Settings Edit
 * @Author: huangtianyang
 * @Date: 2020-03-09 12:16
 */

import { Bind } from 'lodash-decorators';
import { debounce } from 'lodash';
import NavigationContentItem from './NavigationContentItem';
import React from 'react';
import classNames from 'classnames';
import styles from './index.less';

/**
 *
 * @param components {Array,<React.ReactElement>} -组件数组
 * @param componentNames {Array.<String>} -组件描述名
 * @returns {*}
 * @constructor
 */
class NavigationContent extends React.Component {
  constructor(props) {
    super(props);
    this.state = {
      activeIndex: 0,
      windowHeight: window.innerHeight
    };
    this.ref = React.createRef();
    this.scrollTops = [];
    this.nodes = [];
    this.content = null;
    this.disableScrollHandler = false;
  }

  @Bind()
  resizeHandler() {
    if (window.parent === window.self) {
      this.setState({ windowHeight: window.innerHeight });
    }
  }

  componentDidMount() {
    window.addEventListener('resize', this.resizeHandler);
    const node = this.ref.current;
    if (node) {
      const { children } = node;
      this.content = node;
      for (let i = 0; i < children.length; i++) {
        this.nodes.push(children[i]);
        this.scrollTops.push(children[i].offsetTop - node.offsetTop);
      }
    }
  }

  componentWillUnmount() {
    window.removeEventListener('resize', this.resizeHandler);
  }

  @Bind()
  wheelHandler() {
    this.disableScrollHandler = false;
  }

  @Bind()
  getIndex(index) {
    this.setState({ activeIndex: index });
  }

  @Bind()
  getScrollTop(index) {
    return this.nodes[index].offsetTop - this.content.offsetTop;
  }

  render() {
    const { windowHeight, activeIndex } = this.state;
    const { componentNames, components } = this.props;
    return (
      <div
        className={classNames(styles['navigation-content'])}
        style={{ maxHeight: windowHeight - 210 }}
      >
        <div
          ref={this.ref}
          onWheel={this.wheelHandler}
          onScroll={event => {
            if (!this.disableScrollHandler) {
              debounce(e => {
                e.persist();
                const { scrollTop } = this.ref.current;
                for (let i = 0; i < this.scrollTops.length; i++) {
                  const minTop = this.scrollTops[i];
                  const maxTop = this.scrollTops[i + 1];
                  if (
                    (scrollTop >= minTop && maxTop && scrollTop < maxTop) ||
                    (scrollTop >= minTop && !maxTop)
                  ) {
                    this.setState({ activeIndex: i });
                  }
                }
              }, 100)(event);
            }
          }}
          className={styles.content}
        >
          {components.map((component, index) => {
            return (
              <NavigationContentItem
                parentRef={this.ref}
                key={
                  /* eslint-disable-next-line react/no-array-index-key */
                  index
                }
                component={component}
                index={index}
                getIndex={() => this.getIndex(index)}
              />
            );
          })}
        </div>
        <div className={styles.menus}>
          {componentNames.map((name, index) => {
            return (
              <div
                key={name}
                onClick={() => {
                  // eslint-disable-next-line no-undef
                  if (this.ref.current) {
                    this.ref.current.scrollTop = this.getScrollTop(index);
                    this.disableScrollHandler = true;
                    this.setState({ activeIndex: index });
                  }
                }}
                className={classNames(styles.menu, index === activeIndex ? styles.active : null)}
              >
                <div className={classNames(styles.menuCro, index === activeIndex ? styles.active : null,index===componentNames.length-1?styles.menuCroLast:null)} />
                <span>{name}</span>
              </div>
            );
          })}
        </div>
      </div>
    );
  }
}

export default NavigationContent;
